/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.as.repository;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.jboss.as.protocol.StreamUtils;
import org.jboss.as.repository.ContentReference;
import org.jboss.as.repository.DeploymentRepositoryLogger;
import org.jboss.as.repository.DeploymentRepositoryMessages;
import org.jboss.as.repository.HashUtil;
import org.jboss.msc.service.Service;
import org.jboss.msc.service.ServiceName;
import org.jboss.msc.service.ServiceTarget;
import org.jboss.msc.service.StartContext;
import org.jboss.msc.service.StartException;
import org.jboss.msc.service.StopContext;
import org.jboss.vfs.VFS;
import org.jboss.vfs.VirtualFile;

public interface ContentRepository {
    public static final ServiceName SERVICE_NAME = ServiceName.JBOSS.append(new String[]{"content-repository"});
    public static final long OBSOLETE_CONTENT_TIMEOUT = 300000L;
    public static final String DELETED_CONTENT = "deleted-contents";
    public static final String MARKED_CONTENT = "marked-contents";

    public byte[] addContent(InputStream var1) throws IOException;

    public void addContentReference(ContentReference var1);

    public VirtualFile getContent(byte[] var1);

    public boolean hasContent(byte[] var1);

    public boolean syncContent(ContentReference var1);

    public void removeContent(ContentReference var1);

    public Map<String, Set<String>> cleanObsoleteContent();

    public static class Factory {
        public static void addService(ServiceTarget serviceTarget, File repoRoot) {
            ContentRepositoryImpl contentRepository = new ContentRepositoryImpl(repoRoot, 300000L);
            serviceTarget.addService(SERVICE_NAME, (Service)contentRepository).install();
        }

        public static ContentRepository create(File repoRoot) {
            return Factory.create(repoRoot, 300000L);
        }

        static ContentRepository create(File repoRoot, long timeout) {
            return new ContentRepositoryImpl(repoRoot, timeout);
        }

        private static class ContentRepositoryImpl
        implements ContentRepository,
        Service<ContentRepository> {
            protected static final String CONTENT = "content";
            private final File repoRoot;
            protected final MessageDigest messageDigest;
            private final Map<String, Set<ContentReference>> contentHashReferences = new HashMap<String, Set<ContentReference>>();
            private final Map<String, Long> obsoleteContents = new HashMap<String, Long>();
            private final long obsolescenceTimeout;

            protected ContentRepositoryImpl(File repoRoot, long obsolescenceTimeout) {
                if (repoRoot == null) {
                    throw DeploymentRepositoryMessages.MESSAGES.nullVar("repoRoot");
                }
                if (repoRoot.exists()) {
                    if (!repoRoot.isDirectory()) {
                        throw DeploymentRepositoryMessages.MESSAGES.notADirectory(repoRoot.getAbsolutePath());
                    }
                    if (!repoRoot.canWrite()) {
                        throw DeploymentRepositoryMessages.MESSAGES.directoryNotWritable(repoRoot.getAbsolutePath());
                    }
                } else if (!repoRoot.mkdirs()) {
                    throw DeploymentRepositoryMessages.MESSAGES.cannotCreateDirectory(repoRoot.getAbsolutePath());
                }
                this.repoRoot = repoRoot;
                this.obsolescenceTimeout = obsolescenceTimeout;
                try {
                    this.messageDigest = MessageDigest.getInstance("SHA-1");
                }
                catch (NoSuchAlgorithmException e) {
                    throw DeploymentRepositoryMessages.MESSAGES.cannotObtainSha1(e, MessageDigest.class.getSimpleName());
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public byte[] addContent(InputStream stream) throws IOException {
                byte[] sha1Bytes;
                File tmp = File.createTempFile(CONTENT, "tmp", this.repoRoot);
                FileOutputStream fos = new FileOutputStream(tmp);
                MessageDigest messageDigest = this.messageDigest;
                synchronized (messageDigest) {
                    this.messageDigest.reset();
                    try {
                        int read;
                        DigestOutputStream dos = new DigestOutputStream(fos, this.messageDigest);
                        BufferedInputStream bis = new BufferedInputStream(stream);
                        byte[] bytes = new byte[8192];
                        while ((read = bis.read(bytes)) > -1) {
                            dos.write(bytes, 0, read);
                        }
                        fos.flush();
                        fos.getFD().sync();
                        fos.close();
                        fos = null;
                    }
                    finally {
                        ContentRepositoryImpl.safeClose(fos);
                    }
                    sha1Bytes = this.messageDigest.digest();
                }
                File realFile = this.getDeploymentContentFile(sha1Bytes, true);
                if (this.hasContent(sha1Bytes)) {
                    if (!tmp.delete()) {
                        DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteTempFile(tmp.getName());
                        tmp.deleteOnExit();
                    }
                    DeploymentRepositoryLogger.ROOT_LOGGER.debugf("Content was already present in repository at location %s", realFile.getAbsolutePath());
                } else {
                    this.moveTempToPermanent(tmp, realFile);
                    DeploymentRepositoryLogger.ROOT_LOGGER.contentAdded(realFile.getAbsolutePath());
                }
                return sha1Bytes;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void addContentReference(ContentReference reference) {
                Map<String, Set<ContentReference>> map = this.contentHashReferences;
                synchronized (map) {
                    Set<ContentReference> references = this.contentHashReferences.get(reference.getHexHash());
                    if (references == null) {
                        references = new HashSet<ContentReference>();
                        this.contentHashReferences.put(reference.getHexHash(), references);
                    }
                    references.add(reference);
                }
            }

            @Override
            public VirtualFile getContent(byte[] hash) {
                if (hash == null) {
                    throw DeploymentRepositoryMessages.MESSAGES.nullVar("hash");
                }
                return VFS.getChild((URI)this.getDeploymentContentFile(hash, true).toURI());
            }

            @Override
            public boolean syncContent(ContentReference reference) {
                return this.hasContent(reference.getHash());
            }

            @Override
            public boolean hasContent(byte[] hash) {
                return this.getDeploymentContentFile(hash).exists();
            }

            protected File getRepoRoot() {
                return this.repoRoot;
            }

            protected File getDeploymentContentFile(byte[] deploymentHash) {
                return this.getDeploymentContentFile(deploymentHash, false);
            }

            private File getDeploymentContentFile(byte[] deploymentHash, boolean validate) {
                File hashDir = this.getDeploymentHashDir(deploymentHash, validate);
                return new File(hashDir, CONTENT);
            }

            protected File getDeploymentHashDir(byte[] deploymentHash, boolean validate) {
                String sha1 = HashUtil.bytesToHexString(deploymentHash);
                String partA = sha1.substring(0, 2);
                String partB = sha1.substring(2);
                File base = new File(this.getRepoRoot(), partA);
                if (validate) {
                    this.validateDir(base);
                }
                File hashDir = new File(base, partB);
                if (validate && !hashDir.exists() && !hashDir.mkdirs()) {
                    throw DeploymentRepositoryMessages.MESSAGES.cannotCreateDirectory(hashDir.getAbsolutePath());
                }
                return hashDir;
            }

            protected void validateDir(File dir) {
                if (!dir.exists()) {
                    if (!dir.mkdirs()) {
                        throw DeploymentRepositoryMessages.MESSAGES.cannotCreateDirectory(dir.getAbsolutePath());
                    }
                } else {
                    if (!dir.isDirectory()) {
                        throw DeploymentRepositoryMessages.MESSAGES.notADirectory(dir.getAbsolutePath());
                    }
                    if (!dir.canWrite()) {
                        throw DeploymentRepositoryMessages.MESSAGES.directoryNotWritable(dir.getAbsolutePath());
                    }
                }
            }

            private void moveTempToPermanent(File tmpFile, File permanentFile) throws IOException {
                if (!tmpFile.renameTo(permanentFile)) {
                    File localTmp = new File(permanentFile.getParent(), "tmp");
                    try {
                        this.copyFile(tmpFile, localTmp);
                        if (!localTmp.renameTo(permanentFile)) {
                            this.copyFile(localTmp, permanentFile);
                        }
                    }
                    catch (IOException e) {
                        if (permanentFile.exists()) {
                            permanentFile.delete();
                        }
                        throw e;
                    }
                    catch (RuntimeException e) {
                        if (permanentFile.exists()) {
                            permanentFile.delete();
                        }
                        throw e;
                    }
                    finally {
                        if (!tmpFile.delete()) {
                            DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteTempFile(tmpFile.getName());
                            tmpFile.deleteOnExit();
                        }
                        if (localTmp.exists() && !localTmp.delete()) {
                            DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteTempFile(localTmp.getName());
                            localTmp.deleteOnExit();
                        }
                    }
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private void copyFile(File src, File dest) throws IOException {
                FileOutputStream fos = null;
                FileInputStream fis = null;
                try {
                    fos = new FileOutputStream(dest);
                    fis = new FileInputStream(src);
                    StreamUtils.copyStream((InputStream)fis, (OutputStream)fos);
                    fos.flush();
                    fos.getFD().sync();
                    fos.close();
                    fos = null;
                }
                catch (Throwable throwable) {
                    ContentRepositoryImpl.safeClose(fos);
                    ContentRepositoryImpl.safeClose(fis);
                    throw throwable;
                }
                ContentRepositoryImpl.safeClose(fos);
                ContentRepositoryImpl.safeClose(fis);
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void removeContent(ContentReference reference) {
                File grandParent;
                File parent;
                Map<String, Set<ContentReference>> map = this.contentHashReferences;
                synchronized (map) {
                    Set<ContentReference> references = this.contentHashReferences.get(reference.getHexHash());
                    if (references != null) {
                        references.remove(reference);
                        if (!references.isEmpty()) {
                            return;
                        }
                        this.contentHashReferences.remove(reference.getHexHash());
                    }
                }
                File file = this.getDeploymentContentFile(reference.getHash(), true);
                if (!file.delete()) {
                    DeploymentRepositoryLogger.ROOT_LOGGER.contentDeletionError(file.getAbsolutePath());
                }
                if (!(parent = file.getParentFile()).delete()) {
                    DeploymentRepositoryLogger.ROOT_LOGGER.contentDeletionError(parent.getAbsolutePath());
                }
                if ((grandParent = parent.getParentFile()).list().length == 0 && !grandParent.delete()) {
                    DeploymentRepositoryLogger.ROOT_LOGGER.contentDeletionError(grandParent.getAbsolutePath());
                }
                DeploymentRepositoryLogger.ROOT_LOGGER.contentRemoved(file.getAbsolutePath());
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public Map<String, Set<String>> cleanObsoleteContent() {
                HashMap<String, Set<String>> cleanedContents = new HashMap<String, Set<String>>(2);
                cleanedContents.put(ContentRepository.MARKED_CONTENT, new HashSet());
                cleanedContents.put(ContentRepository.DELETED_CONTENT, new HashSet());
                Map<String, Set<ContentReference>> map = this.contentHashReferences;
                synchronized (map) {
                    for (ContentReference fsContent : this.listLocalContents()) {
                        if (!this.contentHashReferences.containsKey(fsContent.getHexHash())) {
                            if (this.markAsObsolete(fsContent)) {
                                ((Set)cleanedContents.get(ContentRepository.DELETED_CONTENT)).add(fsContent.getContentIdentifier());
                                continue;
                            }
                            ((Set)cleanedContents.get(ContentRepository.MARKED_CONTENT)).add(fsContent.getContentIdentifier());
                            continue;
                        }
                        this.obsoleteContents.remove(fsContent.getHexHash());
                    }
                }
                return cleanedContents;
            }

            private boolean markAsObsolete(ContentReference ref) {
                if (this.obsoleteContents.containsKey(ref.getHexHash())) {
                    if (this.obsoleteContents.get(ref.getHexHash()) + this.obsolescenceTimeout < System.currentTimeMillis()) {
                        DeploymentRepositoryLogger.ROOT_LOGGER.obsoleteContentCleaned(ref.getContentIdentifier());
                        this.removeContent(ref);
                        return true;
                    }
                } else {
                    this.obsoleteContents.put(ref.getHexHash(), System.currentTimeMillis());
                }
                return false;
            }

            private Set<ContentReference> listLocalContents() {
                File[] rootHashes;
                HashSet<ContentReference> localReferences = new HashSet<ContentReference>();
                for (File rootHash : rootHashes = this.repoRoot.listFiles()) {
                    if (!rootHash.isDirectory()) continue;
                    File[] complementaryHashes = rootHash.listFiles();
                    if (complementaryHashes == null || complementaryHashes.length == 0) {
                        ContentReference reference = new ContentReference(rootHash.getAbsolutePath(), rootHash.getName());
                        localReferences.add(reference);
                        continue;
                    }
                    for (File complementaryHash : complementaryHashes) {
                        String hash = rootHash.getName() + complementaryHash.getName();
                        ContentReference reference = new ContentReference(complementaryHash.getAbsolutePath(), hash);
                        localReferences.add(reference);
                    }
                }
                return localReferences;
            }

            protected static void safeClose(Closeable closeable) {
                if (closeable != null) {
                    try {
                        closeable.close();
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                }
            }

            public void start(StartContext context) throws StartException {
                DeploymentRepositoryLogger.ROOT_LOGGER.debugf("%s started", ContentRepository.class.getSimpleName());
            }

            public void stop(StopContext context) {
                DeploymentRepositoryLogger.ROOT_LOGGER.debugf("%s stopped", ContentRepository.class.getSimpleName());
            }

            public ContentRepository getValue() throws IllegalStateException, IllegalArgumentException {
                return this;
            }
        }
    }
}

