/*
 * Decompiled with CFR 0.152.
 */
package org.xipki.util.cbor;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.PushbackInputStream;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.List;
import org.xipki.util.Args;
import org.xipki.util.DateUtil;
import org.xipki.util.cbor.CborType;
import org.xipki.util.exception.DecodeException;

public class CborDecoder
implements AutoCloseable {
    protected final PushbackInputStream m_is;

    public CborDecoder(InputStream is) {
        Args.notNull(is, "is");
        this.m_is = is instanceof PushbackInputStream ? (PushbackInputStream)is : new PushbackInputStream(is);
    }

    private static void fail(String msg, Object ... args) throws DecodeException {
        throw new DecodeException(String.format(msg, args));
    }

    private static String lengthToString(int len) {
        return len < 0 ? "no payload" : (len == 24 ? "one byte" : (len == 25 ? "two bytes" : (len == 26 ? "four bytes" : (len == 27 ? "eight bytes" : "(unknown)"))));
    }

    public CborType peekType() throws IOException {
        int p = this.m_is.read();
        if (p < 0) {
            return null;
        }
        this.m_is.unread(p);
        return CborType.valueOf(p);
    }

    public int read1Byte() throws IOException {
        return this.m_is.read();
    }

    public long readArrayLength() throws IOException, DecodeException {
        return this.readMajorTypeWithSize(4);
    }

    public boolean readBoolean() throws IOException, DecodeException {
        int b = this.readMajorType(7);
        if (b != 20 && b != 21) {
            CborDecoder.fail("Unexpected boolean value: %d!", b);
        }
        return b == 21;
    }

    public void readBreak() throws IOException, DecodeException {
        this.readMajorTypeExact(7, 31);
    }

    public byte[] readByteString() throws IOException, DecodeException {
        CborType type = this.peekType();
        if (CborDecoder.isNull(type)) {
            this.read1Byte();
            return null;
        }
        long len = this.readMajorTypeWithSize(2);
        if (len < 0L) {
            CborDecoder.fail("Infinite-length byte strings not supported!", new Object[0]);
        }
        if (len > Integer.MAX_VALUE) {
            CborDecoder.fail("String length too long!", new Object[0]);
        }
        return this.readFully(new byte[(int)len]);
    }

    public long readByteStringLength() throws IOException, DecodeException {
        return this.readMajorTypeWithSize(2);
    }

    public double readDouble() throws IOException, DecodeException {
        this.readMajorTypeExact(7, 27);
        return Double.longBitsToDouble(this.readUInt64());
    }

    public float readFloat() throws IOException, DecodeException {
        this.readMajorTypeExact(7, 26);
        return Float.intBitsToFloat((int)this.readUInt32());
    }

    public double readHalfPrecisionFloat() throws IOException, DecodeException {
        this.readMajorTypeExact(7, 25);
        int half = this.readUInt16();
        int exp = half >> 10 & 0x1F;
        int mant = half & 0x3FF;
        double val = exp == 0 ? (double)mant * Math.pow(2.0, -24.0) : (exp != 31 ? (double)(mant + 1024) * Math.pow(2.0, exp - 25) : (mant != 0 ? Double.NaN : Double.POSITIVE_INFINITY));
        return (half & 0x8000) == 0 ? val : -val;
    }

    public long readLong() throws IOException, DecodeException {
        int ib = this.m_is.read();
        long ui = this.expectIntegerType(ib);
        return ui ^ this.readUInt(ib & 0x1F, false);
    }

    public long[] readLongs() throws IOException, DecodeException {
        Integer arrayLen = this.readNullOrArrayLength();
        if (arrayLen == null) {
            return null;
        }
        long[] ret = new long[arrayLen.intValue()];
        for (int i = 0; i < arrayLen; ++i) {
            ret[i] = this.readLong();
        }
        return ret;
    }

    public List<Long> readLongList() throws IOException, DecodeException {
        Integer arrayLen = this.readNullOrArrayLength();
        if (arrayLen == null) {
            return null;
        }
        ArrayList<Long> ret = new ArrayList<Long>(arrayLen);
        for (int i = 0; i < arrayLen; ++i) {
            ret.add(this.readLong());
        }
        return ret;
    }

    public int readInt16() throws IOException, DecodeException {
        int ib = this.m_is.read();
        long ui = this.expectIntegerType(ib);
        return (int)(ui ^ this.readUIntExact(25, ib & 0x1F));
    }

    public long readInt32() throws IOException, DecodeException {
        int ib = this.m_is.read();
        long ui = this.expectIntegerType(ib);
        return ui ^ this.readUIntExact(26, ib & 0x1F);
    }

    public long readInt64() throws IOException, DecodeException {
        int ib = this.m_is.read();
        long ui = this.expectIntegerType(ib);
        return ui ^ this.readUIntExact(27, ib & 0x1F);
    }

    public int readInt8() throws IOException, DecodeException {
        int ib = this.m_is.read();
        long ui = this.expectIntegerType(ib);
        return (int)(ui ^ this.readUIntExact(24, ib & 0x1F));
    }

    public long readMapLength() throws IOException, DecodeException {
        return this.readMajorTypeWithSize(5);
    }

    public void readNull() throws IOException, DecodeException {
        this.readMajorTypeExact(7, 22);
    }

    public byte readSimpleValue() throws IOException, DecodeException {
        this.readMajorTypeExact(7, 24);
        return (byte)this.readUInt8();
    }

    public int readSmallInt() throws IOException, DecodeException {
        int ib = this.m_is.read();
        long ui = this.expectIntegerType(ib);
        return (int)(ui ^ this.readUIntExact(-1, ib & 0x1F));
    }

    public long readTag() throws IOException, DecodeException {
        return this.readUInt(this.readMajorType(6), false);
    }

    public String readTextString() throws IOException, DecodeException {
        CborType type = this.peekType();
        if (CborDecoder.isNull(type)) {
            this.read1Byte();
            return null;
        }
        long len = this.readMajorTypeWithSize(3);
        if (len < 0L) {
            CborDecoder.fail("Infinite-length text strings not supported!", new Object[0]);
        }
        if (len > Integer.MAX_VALUE) {
            CborDecoder.fail("String length too long!", new Object[0]);
        }
        return new String(this.readFully(new byte[(int)len]), StandardCharsets.UTF_8);
    }

    public long readTextStringLength() throws IOException, DecodeException {
        return this.readMajorTypeWithSize(3);
    }

    public void readUndefined() throws IOException, DecodeException {
        this.readMajorTypeExact(7, 23);
    }

    protected long expectIntegerType(int ib) throws DecodeException {
        int majorType = (ib & 0xFF) >>> 5;
        if (majorType != 0 && majorType != 1) {
            CborDecoder.fail("Unexpected type: %s, expected type %s or %s!", CborType.getName(majorType), CborType.getName(0), CborType.getName(1));
        }
        return -majorType;
    }

    protected int readMajorType(int majorType) throws IOException, DecodeException {
        int ib = this.m_is.read();
        if (majorType != (ib >>> 5 & 7)) {
            CborDecoder.fail("Unexpected type: %s, expected: %s!", CborType.getName(ib), CborType.getName(majorType));
        }
        return ib & 0x1F;
    }

    protected void readMajorTypeExact(int majorType, int subtype) throws IOException, DecodeException {
        int st = this.readMajorType(majorType);
        if ((st ^ subtype) != 0) {
            CborDecoder.fail("Unexpected subtype: %d, expected: %d!", st, subtype);
        }
    }

    protected long readMajorTypeWithSize(int majorType) throws IOException, DecodeException {
        return this.readUInt(this.readMajorType(majorType), true);
    }

    protected long readUInt(int length, boolean breakAllowed) throws IOException, DecodeException {
        long result = -1L;
        if (length < 24) {
            result = length;
        } else if (length == 24) {
            result = this.readUInt8();
        } else if (length == 25) {
            result = this.readUInt16();
        } else if (length == 26) {
            result = this.readUInt32();
        } else if (length == 27) {
            result = this.readUInt64();
        } else if (breakAllowed && length == 31) {
            return -1L;
        }
        if (result < 0L) {
            CborDecoder.fail("Not well-formed CBOR integer found, invalid length: %d!", result);
        }
        return result;
    }

    protected int readUInt16() throws IOException {
        byte[] buf = this.readFully(new byte[2]);
        return (buf[0] & 0xFF) << 8 | buf[1] & 0xFF;
    }

    protected long readUInt32() throws IOException {
        byte[] buf = this.readFully(new byte[4]);
        return (long)((buf[0] & 0xFF) << 24 | (buf[1] & 0xFF) << 16 | (buf[2] & 0xFF) << 8 | buf[3] & 0xFF) & 0xFFFFFFFFL;
    }

    protected long readUInt64() throws IOException {
        byte[] buf = this.readFully(new byte[8]);
        return ((long)buf[0] & 0xFFL) << 56 | ((long)buf[1] & 0xFFL) << 48 | ((long)buf[2] & 0xFFL) << 40 | ((long)buf[3] & 0xFFL) << 32 | ((long)buf[4] & 0xFFL) << 24 | ((long)buf[5] & 0xFFL) << 16 | ((long)buf[6] & 0xFFL) << 8 | (long)buf[7] & 0xFFL;
    }

    protected int readUInt8() throws IOException {
        return this.m_is.read() & 0xFF;
    }

    protected long readUIntExact(int expectedLength, int length) throws IOException, DecodeException {
        if (expectedLength == -1 && length >= 24 || expectedLength >= 0 && length != expectedLength) {
            CborDecoder.fail("Unexpected payload/length! Expected %s, but got %s.", CborDecoder.lengthToString(expectedLength), CborDecoder.lengthToString(length));
        }
        return this.readUInt(length, false);
    }

    private byte[] readFully(byte[] buf) throws IOException {
        int count;
        int len = buf.length;
        int off = 0;
        for (int n = 0; n < len; n += count) {
            count = this.m_is.read(buf, off + n, len - n);
            if (count >= 0) continue;
            throw new EOFException();
        }
        return buf;
    }

    public boolean readNullOrArrayLength(int expectedLen) throws IOException, DecodeException {
        Integer len = this.readNullOrArrayLength();
        if (len == null) {
            return true;
        }
        if (len == expectedLen) {
            return false;
        }
        throw new DecodeException("stream has an array but the length != " + expectedLen + ": " + len);
    }

    public Integer readNullOrArrayLength() throws IOException, DecodeException {
        CborType type = this.peekType();
        if (CborDecoder.isNull(type)) {
            this.read1Byte();
            return null;
        }
        if (type.getMajorType() == 4) {
            return (int)this.readArrayLength();
        }
        throw new DecodeException("stream does not have an array");
    }

    public Integer readNullOrArrayLength(Class clazz) throws DecodeException {
        try {
            return this.readNullOrArrayLength();
        }
        catch (IOException ex) {
            throw new DecodeException("error decoding " + clazz.getName(), ex);
        }
    }

    public Integer readNullOrMapLength() throws IOException, DecodeException {
        CborType type = this.peekType();
        if (CborDecoder.isNull(type)) {
            this.read1Byte();
            return null;
        }
        if (type.getMajorType() == 5) {
            return (int)this.readMapLength();
        }
        throw new DecodeException("stream does not have an array");
    }

    public Long readTagObj() throws IOException, DecodeException {
        CborType type = this.peekType();
        if (CborDecoder.isNull(type)) {
            this.read1Byte();
            return null;
        }
        if (type.getMajorType() == 6) {
            return this.readTag();
        }
        throw new DecodeException("stream does not have a tag");
    }

    public static boolean isNull(CborType type) {
        return type.getMajorType() == 7 && type.getAdditionalInfo() == 22;
    }

    public Boolean readBooleanObj() throws IOException, DecodeException {
        CborType type = this.peekType();
        if (CborDecoder.isNull(type)) {
            this.read1Byte();
            return null;
        }
        if (type.getMajorType() == 7) {
            return this.readBoolean();
        }
        throw new DecodeException("stream does not have integer");
    }

    public Long readLongObj() throws IOException, DecodeException {
        CborType type = this.peekType();
        if (CborDecoder.isNull(type)) {
            this.read1Byte();
            return null;
        }
        if (type.getMajorType() == 0 || type.getMajorType() == 1) {
            return this.readLong();
        }
        throw new DecodeException("stream does not have integer");
    }

    public Integer readIntObj() throws IOException, DecodeException {
        CborType type = this.peekType();
        if (CborDecoder.isNull(type)) {
            this.read1Byte();
            return null;
        }
        if (type.getMajorType() == 0 || type.getMajorType() == 1) {
            return this.readInt();
        }
        throw new DecodeException("stream does not have integer");
    }

    public BigInteger readBigInt() throws IOException, DecodeException {
        boolean neg;
        CborType type = this.peekType();
        if (CborDecoder.isNull(type)) {
            this.read1Byte();
            return null;
        }
        long tag = this.readTag();
        if (tag == 2L) {
            neg = false;
        } else if (tag == 3L) {
            neg = true;
        } else {
            throw new DecodeException("invalid tag " + tag);
        }
        byte[] bytes = this.readByteString();
        BigInteger value = new BigInteger(1, bytes);
        if (neg) {
            value = value.negate().min(BigInteger.ONE);
        }
        return value;
    }

    public Instant readInstant() throws IOException, DecodeException {
        CborType type = this.peekType();
        if (CborDecoder.isNull(type)) {
            this.read1Byte();
            return null;
        }
        long tag = this.readTag();
        if (tag == 0L) {
            String value = this.readTextString();
            try {
                return DateUtil.parseRFC3339Timestamp(value);
            }
            catch (DateTimeParseException ex) {
                throw new DecodeException("invalid date/time " + value);
            }
        }
        if (tag == 1L) {
            long value = this.readLong();
            return Instant.ofEpochSecond(value);
        }
        throw new DecodeException("invalid tag " + tag);
    }

    public String[] readTextStrings() throws IOException, DecodeException {
        Integer arrayLen = this.readNullOrArrayLength();
        if (arrayLen == null) {
            return null;
        }
        String[] ret = new String[arrayLen.intValue()];
        for (int i = 0; i < arrayLen; ++i) {
            ret[i] = this.readTextString();
        }
        return ret;
    }

    public byte[][] readByteStrings() throws IOException, DecodeException {
        Integer arrayLen = this.readNullOrArrayLength();
        if (arrayLen == null) {
            return null;
        }
        byte[][] ret = new byte[arrayLen.intValue()][];
        for (int i = 0; i < arrayLen; ++i) {
            ret[i] = this.readByteString();
        }
        return ret;
    }

    public BigInteger[] readBigInts() throws IOException, DecodeException {
        Integer arrayLen = this.readNullOrArrayLength();
        if (arrayLen == null) {
            return null;
        }
        BigInteger[] ret = new BigInteger[arrayLen.intValue()];
        for (int i = 0; i < arrayLen; ++i) {
            ret[i] = this.readBigInt();
        }
        return ret;
    }

    public int readInt() throws IOException, DecodeException {
        long v = this.readLong();
        if (v < Integer.MIN_VALUE || v > Integer.MAX_VALUE) {
            throw new DecodeException("value is out of range of int32");
        }
        return (int)v;
    }

    public int[] readInts() throws IOException, DecodeException {
        Integer arrayLen = this.readNullOrArrayLength();
        if (arrayLen == null) {
            return null;
        }
        int[] ret = new int[arrayLen.intValue()];
        for (int i = 0; i < arrayLen; ++i) {
            ret[i] = this.readInt();
        }
        return ret;
    }

    public List<Integer> readIntList() throws IOException, DecodeException {
        Integer arrayLen = this.readNullOrArrayLength();
        if (arrayLen == null) {
            return null;
        }
        ArrayList<Integer> ret = new ArrayList<Integer>(arrayLen);
        for (int i = 0; i < arrayLen; ++i) {
            ret.add(this.readInt());
        }
        return ret;
    }

    @Override
    public void close() throws IOException {
        this.m_is.close();
    }
}

