/*
 * Copyright 2010 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package com.fusesource.fmc.restjmx.model;

import java.net.URLEncoder;

import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlRootElement;

import com.fusesource.fmc.restjmx.util.JMXNamespaceHelper;

/**
 * Represents a MBean resource
 * 
 * @version $Revision: 1.0 $
 */
@XmlRootElement(name = "mbean")
@XmlAccessorType(XmlAccessType.FIELD)
public class MBean {
    @XmlAttribute
    private String objectName;
    @XmlAttribute
    private String qualifiedObjectName;
    @XmlAttribute
    private String objectClass;
    @XmlAttribute
    private String uriPath;
    @XmlAttribute
    private String namespace;

    public MBean() {
    }

    public MBean(String parentURIPath, String namespace, ObjectInstance objectInstance) throws Exception {
        this();
        this.objectClass = objectInstance.getClassName();
        this.namespace = namespace;

        ObjectName unqualifiedObjectName = objectInstance.getObjectName();
        String containingNamespace = JMXNamespaceHelper.getContainingNamespace(unqualifiedObjectName);
        boolean isNamespaceMBean = JMXNamespaceHelper.isNamespaceObjectName(unqualifiedObjectName);

        if (isNamespaceMBean) {
            if (containingNamespace.equals(namespace)) {
                // e.g. "foo//foo//:type=JMXNamespace" where namespace = "foo"
                unqualifiedObjectName = JMXNamespaceHelper.removePath(containingNamespace,
                                                                      unqualifiedObjectName);
                this.uriPath = "/jmx/namespaces/" + namespace + "/mbeans";
                this.objectName = unqualifiedObjectName.getCanonicalName();
            } else {
                // e.g. "foo//:type=JMXNamespace"
                this.uriPath = "/jmx/namespaces/" + JMXNamespaceHelper.getNamespace(unqualifiedObjectName)
                             + "/mbeans";
                this.objectName = unqualifiedObjectName.getCanonicalName();
            }
        } else {
            if (containingNamespace.length() == 0) {
                // e.g. "a:x=y"
                this.uriPath = parentURIPath + '/'
                             + URLEncoder.encode(unqualifiedObjectName.getCanonicalName(), "UTF-8");
                this.objectName = unqualifiedObjectName.getCanonicalName();
            } else {
                // e.g. "foo//a:x=y"
                unqualifiedObjectName = JMXNamespaceHelper.removePath(containingNamespace,
                                                                      unqualifiedObjectName);
                this.uriPath = "/jmx/namespaces/" + containingNamespace + "/mbeans/"
                             + URLEncoder.encode(unqualifiedObjectName.getCanonicalName(), "UTF-8");
                this.objectName = unqualifiedObjectName.getCanonicalName();
            }
        }

        this.namespace = namespace;
    }

    @Override
    public String toString() {
        return "MBean{namespace='" + this.namespace + "' name='" + this.objectName + "' uriPath='"
                + getUriPath() + "'}";
    }

    public String getObjectName() {
        return this.objectName;
    }

    public String getQualifiedObjectName() {
        return this.qualifiedObjectName;
    }

    public String getObjectClass() {
        return this.objectClass;
    }

    public String getUriPath() {
        return this.uriPath;
    }

    public String getNamespace() {
        return this.namespace;
    }
}
