/*
 * Copyright 2010 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package com.fusesource.fmc.restjmx.resources;

import java.lang.management.ManagementFactory;
import java.lang.reflect.Constructor;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;
import javax.management.Attribute;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;

import com.sun.jersey.api.representation.Form;
import com.sun.jersey.api.view.ImplicitProduces;
import com.fusesource.fmc.restjmx.util.JMXNamespaceHelper;


/**
 * The resource for a MBean
 *
 * @version $Revision: 1.0 $
 */
@ImplicitProduces(Constants.HTML_MIME_TYPES)
public class AttributeResource {
    private static final Map<String, Class<?>> PRIMITIVE_TYPE_MAP = new HashMap<String, Class<?>>();
    
    private String uriPath;
    private String namespace;
    private ObjectName objectName;
    private MBeanAttributeInfo attributeInfo;
    private String value;
    
    static {
        PRIMITIVE_TYPE_MAP.put(boolean.class.getName(), Boolean.class);
        PRIMITIVE_TYPE_MAP.put(byte.class.getName(), Byte.class);
        PRIMITIVE_TYPE_MAP.put(char.class.getName(), Character.class);
        PRIMITIVE_TYPE_MAP.put(short.class.getName(), Short.class);
        PRIMITIVE_TYPE_MAP.put(int.class.getName(), Integer.class);
        PRIMITIVE_TYPE_MAP.put(long.class.getName(), Long.class);
        PRIMITIVE_TYPE_MAP.put(float.class.getName(), Float.class);
        PRIMITIVE_TYPE_MAP.put(double.class.getName(), Double.class);
    }

    public AttributeResource(String uriPath, 
                             String namespace, 
                             ObjectName objectName, 
                             MBeanAttributeInfo attributeInfo, 
                             Object value) {
        this.uriPath = uriPath;
        this.namespace = namespace;
        this.objectName = objectName;
        this.attributeInfo = attributeInfo;
        this.value = value.toString();
    }

    /**
     * Returns this MBean
     */
    @GET
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public com.fusesource.fmc.restjmx.model.Attribute getAttribute() throws Exception {
        return new com.fusesource.fmc.restjmx.model.Attribute(uriPath, attributeInfo, value);
    }

    /**
     * Posts an update to the attribute value with the payload being text, XML or JSON
     */
    @POST
    @Consumes({ MediaType.TEXT_PLAIN, MediaType.TEXT_HTML, MediaType.TEXT_XML, MediaType.APPLICATION_XML })
    public Response postMessage(@Context UriInfo uriInfo, @Context HttpHeaders headers,  final String val) 
        throws Exception {
        
        return setAttributeAndRedirect(uriInfo, headers, val);
    }

    /**
     * Posts an update to the attribute value taking the form data and extracting the <code>body</code> 
     * field as the new value.
     */
    @POST
    @Consumes("application/x-www-form-urlencoded")
    public Response postMessageForm(@Context UriInfo uriInfo, @Context HttpHeaders headers, Form formData) 
        throws Exception {
        return setAttributeAndRedirect(uriInfo, headers, formData.getFirst("value", String.class));
    }

    protected Response setAttributeAndRedirect(UriInfo uriInfo, HttpHeaders headers, String val) 
        throws Exception {
        MBeanServerConnection mBeanServerConnection = ManagementFactory.getPlatformMBeanServer();

        ObjectName qualifiedObjectName = this.namespace == null 
                                         ? this.objectName 
                                         : JMXNamespaceHelper.insertPath(this.namespace, this.objectName);
        Attribute attribute = new Attribute(this.attributeInfo.getName(), toAttributeType(val));

        mBeanServerConnection.setAttribute(qualifiedObjectName, attribute);
        
        URI uri = uriInfo.getRequestUriBuilder().build();
        return Response.seeOther(uri).build();
    }

    /**
     * Converts the string value to the type specified by the attribute info
     */
    private Object toAttributeType(String val) throws Exception {
        String type = this.attributeInfo.getType();
        Class<?> clazz = null;
        try {
            clazz = Class.forName(type, false, this.getClass().getClassLoader());
        } catch (ClassNotFoundException e) {
            // if is it a primitive, then map to the class equivalent
            clazz = PRIMITIVE_TYPE_MAP.get(type);
            if (clazz == null) {
                throw e;
            }
        }
        
        Constructor<?> constructor = clazz.getConstructor(new Class[] { String.class });
        return constructor.newInstance(new Object[] { val });
    }

    // Properties
    //-------------------------------------------------------------------------

    public String getUriPath() {
        return this.uriPath;
    }

    public String getNamespace() {
        return this.namespace;
    }

    public String getObjectName() {
        return this.objectName.getCanonicalName();
    }

    public String getName() {
        return this.attributeInfo.getName();
    }

    public String getType() {
        return this.attributeInfo.getType();
    }

    public String getDescription() {
        return this.attributeInfo.getDescription();
    }

    public boolean getIsWritable() {
        return this.attributeInfo.isWritable();
    }

    public String getValue() {
        return this.value;
    }
}
