/*
 * Copyright 2010 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package com.fusesource.fmc.restjmx.resources;

import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.management.AttributeList;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriInfo;

import com.sun.jersey.api.view.ImplicitProduces;
import com.fusesource.fmc.restjmx.model.Attribute;
import com.fusesource.fmc.restjmx.model.Attributes;
import com.fusesource.fmc.restjmx.util.UriInfoHelper;

/**
 * Represents the MBeans resource
 *
 * @version $Revision: 1.0 $
 */
@ImplicitProduces(Constants.HTML_MIME_TYPES)
public class AttributesResource {
    
    private String uriPath;
    private String namespace;
    private ObjectName objectName;
    private List<String> attributeNames;
    
    public AttributesResource(String uriPath, 
                              String namespace, 
                              ObjectName objectName, 
                              List<String> attributeNames) {
        this.uriPath = uriPath;
        this.namespace = namespace;
        this.objectName = objectName;
        this.attributeNames = attributeNames;
    }

    public Map<String, Object[]> getAttributeMap() throws Exception {
        Map<String, Object[]> mBeanAttributeMap = new TreeMap<String, Object[]>();
        
        MBeanServerConnection mBeanServerConnection = ManagementFactory.getPlatformMBeanServer();
        MBeanInfo mBeanInfo = mBeanServerConnection.getMBeanInfo(objectName);
        MBeanAttributeInfo[] mBeanAttributeInfos = mBeanInfo.getAttributes();
        
        if (attributeNames == null) {
            attributeNames = new ArrayList<String>();
            for (int i = 0; i < mBeanAttributeInfos.length; i++) {
                attributeNames.add(mBeanAttributeInfos[i].getName());
            }
        }
        
        AttributeList attributeList = mBeanServerConnection.getAttributes(
                objectName, attributeNames.toArray(new String[attributeNames.size()]));
        
        for (Object attribute : attributeList) {
            String name = ((javax.management.Attribute)attribute).getName();
            Object value = ((javax.management.Attribute)attribute).getValue();
            for (int i = 0; i < mBeanAttributeInfos.length; i++) {
                if (mBeanAttributeInfos[i].getName().equals(name)) {
                    mBeanAttributeMap.put(name, new Object[] { mBeanAttributeInfos[i], value });
                    break;
                }
            }
        }
        
        return mBeanAttributeMap;
    }


    /**
     * Returns the resource for an individual attribute
     *
     * @param info
     * @param attributeName The attribute name
     */
    @Path("{attributeName}")
    public AttributeResource getAttributeResource(@Context UriInfo info, 
                                                  @PathParam("attributeName") String attributeName) 
        throws Exception {
        Object[] attributeData = getAttributeMap().get(attributeName);
        return new AttributeResource(UriInfoHelper.getUriPath(info, attributeName), 
                                     namespace, 
                                     objectName, 
                                     (MBeanAttributeInfo)attributeData[0], 
                                     attributeData[1]);
    }
    
    /**
     * Returns a list of the active MBeans available
     */
    @GET
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public Attributes getAttributes() throws Exception {
        return new Attributes(uriPath, getAttributeMap());
    }

    // Properties
    //-------------------------------------------------------------------------

    public String getNamespace() throws Exception {
        return namespace == null ? "" : namespace;
    }

    public String getObjectName() {
        return objectName.getCanonicalName();
    }

    public List<Attribute> getAttributeList() throws Exception {
        return getAttributes().getAttributes();
    }
    
    public String getFilter() {
        return attributeNames == null ? "" : attributeNames.toString();
    }
}
