/*
 * Copyright 2010 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package com.fusesource.fmc.restjmx.resources;

import java.lang.management.ManagementFactory;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import javax.management.MBeanServerConnection;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriInfo;

import com.sun.jersey.api.view.ImplicitProduces;
import com.fusesource.fmc.restjmx.model.MBean;
import com.fusesource.fmc.restjmx.model.MBeans;
import com.fusesource.fmc.restjmx.util.UriInfoHelper;

/**
 * Represents the MBeans resource
 *
 * @version $Revision: 1.0 $
 */
@ImplicitProduces(Constants.HTML_MIME_TYPES)
public class MBeansResource {
    
    private String uriPath;
    private String pattern;
    private String namespace;
    
    public MBeansResource(String uriPath, String pattern, String namespace) {
        this.uriPath = uriPath;
        this.pattern = pattern;
        this.namespace = namespace;
    }

    @SuppressWarnings("unchecked")
    public Map<String, ObjectInstance> getMBeanMap() throws Exception {
        Map<String, ObjectInstance> mBeanMap = new TreeMap<String, ObjectInstance>();
        
        ObjectName queryName = null;
        if (pattern != null) {
            queryName = new ObjectName(pattern);
        }
        MBeanServerConnection mBeanServerConnection = ManagementFactory.getPlatformMBeanServer();
        Set<ObjectInstance> mBeanInstances = mBeanServerConnection.queryMBeans(queryName, null);
        for (ObjectInstance mBeanInstance : mBeanInstances) {
            mBeanMap.put(mBeanInstance.getObjectName().getCanonicalName(), mBeanInstance);
        }
        
        return mBeanMap;
    }

    /**
     * Returns the resource for an individual MBean
     *
     * @param objectName The canonical form of the MBean ObjectName
     */
    @Path("{objectName}")
    public MBeanResource getMBeanResource(@Context UriInfo info, @PathParam("objectName") String objectName) 
        throws Exception {
        
        return new MBeanResource(UriInfoHelper.getUriPath(info, objectName), 
                                 namespace, 
                                 new ObjectName(objectName),
                                 getMBeanMap().get(objectName));
    }
    
    /**
     * Returns a list of the active MBeans available
     */
    @GET
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public MBeans getMBeans() throws Exception {
        return new MBeans(uriPath, namespace, getMBeanMap());
    }

    // Properties
    //-------------------------------------------------------------------------

    public List<MBean> getMBeanList() throws Exception {
        return getMBeans().getMBeans();
    }

    public String getFilter() throws Exception {
        return pattern == null ? "" : pattern;
    }

    public String getNamespace() throws Exception {
        return namespace == null ? "" : namespace;
    }
}
