/*
 * Copyright 2010 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package com.fusesource.fmc.restjmx.resources;

import java.lang.management.ManagementFactory;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriInfo;

import com.sun.jersey.api.view.ImplicitProduces;
import com.fusesource.fmc.restjmx.model.Namespace;
import com.fusesource.fmc.restjmx.model.Namespaces;
import com.fusesource.fmc.restjmx.util.JMXNamespaceHelper;
import com.fusesource.fmc.restjmx.util.UriInfoHelper;

/**
 * Represents the Namespaces resource
 *
 * @version $Revision: 1.0 $
 */
@ImplicitProduces(Constants.HTML_MIME_TYPES)
public class NamespacesResource {
    
    private String uriPath;
    private List<String> namespaceNames;
    
    public NamespacesResource(String uriPath, List<String> namespaceNames) {
        this.uriPath = uriPath;
        this.namespaceNames = namespaceNames;
    }

    @SuppressWarnings("unchecked")
    public Map<String, String> getNamespaceMap() throws Exception {
        Map<String, String> namespaceMap = new TreeMap<String, String>();
        
        ObjectName queryName = new ObjectName("*" + JMXNamespaceHelper.NAMESPACE_SEPARATOR
                                              + JMXNamespaceHelper.NAMESPACE_OBJECTNAME_SUFFIX);
        
        MBeanServerConnection mBeanServerConnection = ManagementFactory.getPlatformMBeanServer();
        Set<ObjectName> objectNames = mBeanServerConnection.queryNames(queryName, null);
        for (ObjectName objectName : objectNames) {
            String domain = objectName.getDomain();
            String namespace = domain.substring(0, domain.indexOf(JMXNamespaceHelper.NAMESPACE_SEPARATOR));
            namespaceMap.put(namespace, namespace);
        }
        
        return namespaceMap;
    }

    /**
     * Returns the resource for an individual JMX namespace
     *
     * @param namespace The JMX namespace name
     */
    @Path("{namespace}")
    public NamespaceResource getMBeanResource(@Context UriInfo info, 
                                              @PathParam("namespace") String namespace) 
        throws Exception {
        
        return new NamespaceResource(UriInfoHelper.getUriPath(info, namespace), namespace);
    }
    
    /**
     * Returns a list of the active namespaces available
     */
    @GET
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public Namespaces getNamespaces() throws Exception {
        return new Namespaces(uriPath, getNamespaceMap());
    }

    // Properties
    //-------------------------------------------------------------------------

    public List<Namespace> getNamespaceList() throws Exception {
        return getNamespaces().getNamespaces();
    }
    
    public String getFilter() {
        return namespaceNames == null ? "" : namespaceNames.toString();
    }
}
