/*
 * Copyright 2010 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package com.fusesource.fmc.restjmx.util;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

public final class JMXNamespaceHelper {

    public static final String NAMESPACE_SEPARATOR = "//";
    public static final String NAMESPACE_OBJECTNAME_SUFFIX = ":type=JMXNamespace";
    public static final String NAMESPACE_OBJECTNAME = NAMESPACE_SEPARATOR + NAMESPACE_OBJECTNAME_SUFFIX;

    private JMXNamespaceHelper() {
    }

    public static String getContainingNamespace(ObjectName objectName) {
        if (objectName.getCanonicalName().endsWith(NAMESPACE_OBJECTNAME)) {
            String domain = objectName.getDomain();

            // the next test deals with a name like "foo//bar//:type=JMXNamespace", such that we only
            // don't return a namespace if there is only one level of namespace
            // (e.g. "foo//:type=JMXNamespace")
            if (domain.indexOf(NAMESPACE_SEPARATOR) == domain.lastIndexOf(NAMESPACE_SEPARATOR)) {
                return "";
            }
        }

        return getNamespace(objectName);
    }

    public static String getNamespace(ObjectName objectName) {
        String domain = objectName.getDomain();
        int index = domain.indexOf(NAMESPACE_SEPARATOR);

        if (index < 0) {
            return "";
        }
        String jmxNamespace = domain.substring(0, index);
        if (jmxNamespace.indexOf('*') > -1 || jmxNamespace.indexOf('?') > -1) {
            return "";
        }
        return jmxNamespace;
    }

    public static ObjectName getNamespaceObjectName(String jmxNamespace) {
        try {
            jmxNamespace += JMXNamespaceHelper.NAMESPACE_SEPARATOR;
            return new ObjectName(jmxNamespace + NAMESPACE_OBJECTNAME_SUFFIX);
        } catch (MalformedObjectNameException e) {
            throw new RuntimeException(e);
        }
    }

    public static boolean isNamespaceObjectName(ObjectName objectName) {
        return objectName.getCanonicalName().endsWith(NAMESPACE_OBJECTNAME);
    }

    public static ObjectName insertPath(String jmxNamespace, ObjectName objectName) {
        if (jmxNamespace.startsWith("/")) {
            throw new IllegalArgumentException("Namespace cannot begin with '/'");
        }
        if (jmxNamespace.endsWith("/")) {
            throw new IllegalArgumentException("Namespace cannot end with '/'");
        }

        String canonicalName = objectName.getCanonicalName();
        jmxNamespace += JMXNamespaceHelper.NAMESPACE_SEPARATOR;

        if (canonicalName.startsWith("/")) {
            throw new IllegalArgumentException("ObjectNames cannot start with '/'");
        }
        try {
            return new ObjectName(jmxNamespace + canonicalName);
        } catch (MalformedObjectNameException e) {
            throw new RuntimeException(e);
        }
    }

    public static ObjectName removePath(String jmxNamespace, ObjectName objectName) {
        if (jmxNamespace == null || jmxNamespace.length() == 0) {
            return objectName;
        }
        if (jmxNamespace.startsWith("/")) {
            throw new IllegalArgumentException("Namespace cannot begin with '/'");
        }
        if (jmxNamespace.endsWith("/")) {
            throw new IllegalArgumentException("Namespace cannot end with '/'");
        }

        String canonicalName = objectName.getCanonicalName();
        jmxNamespace += JMXNamespaceHelper.NAMESPACE_SEPARATOR;

        if (canonicalName.startsWith("/")) {
            throw new IllegalArgumentException("ObjectNames cannot start with '/'");
        }
        ObjectName namespaceObjectName = JMXNamespaceHelper.getNamespaceObjectName(jmxNamespace);
        if (objectName.equals(namespaceObjectName)) {
            return objectName;
        }
        try {
            return new ObjectName(canonicalName.substring(jmxNamespace.length()));
        } catch (MalformedObjectNameException e) {
            throw new RuntimeException(e);
        }
    }

}
