/*
 * Copyright 2010 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package com.fusesource.fmc.restjmx.util;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import javax.ws.rs.core.PathSegment;
import javax.ws.rs.core.UriInfo;

public final class UriInfoHelper {
    private UriInfoHelper() {
    }
    
    /**
     * Equivalent to calling getUriPath(info, endingSegmentPath, true).
     * 
     * @see #getUriPath(UriInfo, String, boolean)
     */
    public static String getUriPath(UriInfo info, String endingSegmentPath) {
        return getUriPath(info, endingSegmentPath, true);
    }
    
    /**
     * Assuming the endingSegmentPath is unique within the given URI, compose a
     * URI up to and including the endingSegmentPath.
     * <p>
     * For example getUriPath(UriInfo info, String endingSegmentPath) where
     * info.getPath() = "x/y/z" and endingSegmentPath = "y", would yield "x/y".
     * 
     * @param info The source URI.
     * @param endingSegmentPath The path used as the test for the last segment.
     * to be included composed URI
     * @param decode Controls whether sequences of escaped octets in path segments and matrix parameter 
     *               values are decoded (true) or not (false).
     * 
     * @return The composed URI 
     */
    public static String getUriPath(UriInfo info, String endingSegmentPath, boolean decode) {
        try {
            StringBuffer path = new StringBuffer();
            for (PathSegment pathSegment : info.getPathSegments(decode)) {
                path.append('/').append(URLEncoder.encode(pathSegment.getPath(), "UTF-8"));
                if (pathSegment.getPath().equals(endingSegmentPath)) {
                    break;
                }
            }

            return path.toString();
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }
}
