/*
 * Copyright 2012 FuseSource, Corp. All rights reserved.
 * http://fusesource.com
 * 
 * The software in this package is published under the terms of the
 * Fuse Custom Distribution Creator 1.0 User License Agreement.
 * A copy has been included with this distribution in the
 * fusecdc-license.txt file.
 */
package com.fusesource.tooling.fuse.cdc;

import com.fusesource.tooling.fuse.cdc.api.Configuration;
import com.fusesource.tooling.fuse.cdc.api.CustomDistributionCreator;
import com.fusesource.tooling.fuse.cdc.api.aether.RepositorySystemFactory;
import com.fusesource.tooling.fuse.cdc.api.aether.RepositorySystemSessionFactory;
import com.fusesource.tooling.fuse.cdc.api.resolver.RepositoryConfiguration;
import com.fusesource.tooling.fuse.cdc.api.resolver.ResolverUtils;
import com.fusesource.tooling.fuse.cdc.api.utils.ObjectUtils;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonatype.aether.RepositorySystem;
import org.sonatype.aether.RepositorySystemSession;
import org.sonatype.aether.artifact.Artifact;
import org.sonatype.aether.repository.RemoteRepository;
import org.sonatype.aether.repository.RepositoryPolicy;
import org.sonatype.aether.util.artifact.DefaultArtifact;

/**
 * Command line interface for the Fuse {@link CustomDistributionCreator}.
 */
public class FuseCDCCLI {

    public static final String ORIG_DISTRO_KEY = "original.distribution.artifact";
    public static final String FEATURE_DESCRIPTOR_KEY = "feature.descriptors";
    public static final String FEATURE_NAMES_KEY = "features.names";

    private static final Logger LOGGER = LoggerFactory.getLogger(FuseCDCCLI.class);
    private static final String USAGE = "[-p]";
    private static final String HEADER = "FuseCDC - Fuse Custom Distribution Creator, Copyright 2012 FuseSource, Corp. All rights reserved.";
    private static final String FOOTER = "For more details, see our website at: http://fusesource.com";

    private String outputDirectory = "build";
    private final List<String> defaultFeatures;

    public FuseCDCCLI() {
        defaultFeatures = new ArrayList<String>();
        defaultFeatures.add("karaf-framework");
        defaultFeatures.add("jasypt-encryption");
        defaultFeatures.add("config");
        defaultFeatures.add("management");
        defaultFeatures.add("fabric-boot-commands");
        defaultFeatures.add("fabric-bundle");
        defaultFeatures.add("patch");
    }

    public void execute(String propertyFileName) throws Exception {

        Properties props = loadProperties(propertyFileName);

        for (Object objKey : props.keySet()) {
            String key = (String)objKey;
            if (key.startsWith("maven.wagon")) {
                String value = props.getProperty(key);
                System.setProperty(key, value);
            }
        }

        // time in ms (default 1800000) read time out.
        if (System.getProperty("maven.wagon.rto") == null) {
            System.setProperty("maven.wagon.rto", "30000");
        }
        Configuration c = new Configuration();

        Artifact distro = new DefaultArtifact(ResolverUtils.toArtifactCoords(props.getProperty(ORIG_DISTRO_KEY)));
        c.setDistributionArtifact(distro);

        if (props.containsKey(FEATURE_DESCRIPTOR_KEY)) {
            List<String> featureDescriptors = new ArrayList<String>();
            String[] features = ((String)props.get(FEATURE_DESCRIPTOR_KEY)).split(",");
            for (String feature : features) {
                featureDescriptors.add(feature);
            }
            c.setDescriptors(featureDescriptors);
        }

        List<String> featureNames = new ArrayList<String>();

        if (props.containsKey(FEATURE_NAMES_KEY)) {
            if (ObjectUtils.isStringEmpty(FEATURE_NAMES_KEY)) {
                featureNames.add("default");
            } else if (props.containsKey("all")) {
                // handle all
            } else {
                featureNames.addAll(defaultFeatures);
                String[] features = ((String)props.get(FEATURE_NAMES_KEY)).split(",");
                for (String feature : features) {
                    if (!featureNames.contains(feature)) {
                        featureNames.add(feature);
                    }
                }
            }
        }

        c.setFeatures(featureNames);

        if (ObjectUtils.isStringNotEmpty(props.getProperty("build.output.directory"))) {
            c.setBuildDirectory(new File(props.getProperty("build.output.directory")));
        } else {
            c.setBuildDirectory(new File(getOutputDirectory()));
        }

        if (ObjectUtils.isStringNotEmpty(props.getProperty("distribution.create"))) {
            c.setRepackage(Boolean.parseBoolean(props.getProperty("distribution.create")));

            if (ObjectUtils.isStringNotEmpty(props.getProperty("distribution.output.directory"))) {
                c.setDistributionDirectory(new File(props.getProperty("distribution.output.directory")));
            } else {
                c.setDistributionDirectory(new File("dist"));
            }
            if (ObjectUtils.isStringNotEmpty(props.getProperty("distribution.output.final.name"))) {
                c.setRepackagedFileName(props.getProperty("distribution.output.final.name"));
            } else {
                String newOutputFileName = propertyFileName.substring(0, propertyFileName.lastIndexOf("."));
                newOutputFileName = newOutputFileName.replace('.', '-');
                c.setRepackagedFileName(newOutputFileName);
            }

        } else {
            c.setRepackage(false);
        }

        RepositorySystem system = RepositorySystemFactory.newRepositorySystem();
        c.setRepositorySystem(system);

        RepositorySystemSession session = null;
        String localRepo = props.getProperty("local.repository");
        if (ObjectUtils.isStringNotEmpty(localRepo)) {
            File localRepoDirectory = new File(localRepo);
            if (!localRepoDirectory.exists()) {
                localRepoDirectory.mkdirs();
            }

            if (localRepoDirectory.exists() && new File(localRepo).canWrite()) {
                session = RepositorySystemSessionFactory.newRepositorySystemSession(system, localRepo);
            } else {
                throw new Exception("Unable to create the local repository defined by property local.repository: " + localRepo);
            }
        } else {
            session = RepositorySystemSessionFactory.newRepositorySystemSession(system, "repository");
        }
        c.setRepositorySystemSession(session);

        List<RemoteRepository> remoteRepositories = new ArrayList<RemoteRepository>();
        int repositoryKeyNumber = 1;
        String repositoryKey = "repository." + repositoryKeyNumber;
        while (props.containsKey(repositoryKey)) {
            RepositoryConfiguration rc = createRepositoryConfiguration(props, repositoryKeyNumber);
            remoteRepositories.add(ResolverUtils.createRepository(rc));
            repositoryKey = "repository." + ++repositoryKeyNumber;
        }
        c.setRemoteRepositories(remoteRepositories);

        CustomDistributionCreator dc = new CustomDistributionCreator(c);
        try {
            dc.execute();
        } catch (Exception e) {
            LOGGER.error("Exception", e);
        }
    }

    private RepositoryConfiguration createRepositoryConfiguration(Properties props, int repositoryKeyNumber) throws Exception {
        RepositoryConfiguration rc = new RepositoryConfiguration();

        String url = props.getProperty("repository." + repositoryKeyNumber + ".url");
        if (ObjectUtils.isStringNotEmpty(url)) {
            // Hack to ensure the URL that is defined is valid
            try {
                new URL(url);
            } catch (Exception e) {
                throw new Exception(e.getLocalizedMessage(), e);
            }
            rc.setUrl(url);
        } else {
            throw new Exception("The repository." + repositoryKeyNumber + ".url property is empty");
        }

        // If the URL is OK we can guess the rest
        String id = props.getProperty("repository." + repositoryKeyNumber);
        if (ObjectUtils.isStringNotEmpty(id)) {
            rc.setId(id);
        } else {
            rc.setId("repository." + repositoryKeyNumber);
        }

        String layout = props.getProperty("repository." + repositoryKeyNumber + ".layout");
        if (ObjectUtils.isStringNotEmpty(layout)) {
            if (layout.equalsIgnoreCase("default") || layout.equalsIgnoreCase("legacy")) {
                rc.setLayout(layout);
            } else {
                throw new Exception(String.format("Repository layout type %s is not supported", layout));
            }
        }

        String username = props.getProperty("repository." + repositoryKeyNumber + ".username", null);
        rc.setUsername(username);
        String password = props.getProperty("repository." + repositoryKeyNumber + ".password", null);
        rc.setPassword(password);

        String release = props.getProperty("repository." + repositoryKeyNumber + ".release.enabled", "true");
        if (ObjectUtils.isStringNotEmpty(release)) {
            if (release.equalsIgnoreCase("true")) {
                rc.setReleaseEnabled(true);

                String releaseUpdatePolicy = props.getProperty("repository." + repositoryKeyNumber + ".release.policy.update", "daily");
                if (ObjectUtils.isStringNotEmpty(releaseUpdatePolicy)) {
                    if (releaseUpdatePolicy.equalsIgnoreCase("daily")) {
                        rc.setReleaseUpdatePolicy(RepositoryPolicy.UPDATE_POLICY_DAILY);
                    } else if (releaseUpdatePolicy.equalsIgnoreCase("always")) {
                        rc.setReleaseUpdatePolicy(RepositoryPolicy.UPDATE_POLICY_ALWAYS);
                    } else if (releaseUpdatePolicy.equalsIgnoreCase("never")) {
                        rc.setReleaseUpdatePolicy(RepositoryPolicy.UPDATE_POLICY_NEVER);
                    } else {
                        LOGGER.warn("{} is not a supported repository update policy.  Please use always, daily or never.", releaseUpdatePolicy);
                    }
                }

                String releaseChecksumPolicy = props.getProperty("repository." + repositoryKeyNumber + ".release.policy.checksum", "warn");
                if (ObjectUtils.isStringNotEmpty(releaseChecksumPolicy)) {
                    if (releaseChecksumPolicy.equalsIgnoreCase("fail")) {
                        rc.setReleaseChecksumPolicy(RepositoryPolicy.CHECKSUM_POLICY_FAIL);
                    } else if (releaseChecksumPolicy.equalsIgnoreCase("ignore")) {
                        rc.setReleaseChecksumPolicy(RepositoryPolicy.CHECKSUM_POLICY_IGNORE);
                    } else if (releaseChecksumPolicy.equalsIgnoreCase("warn")) {
                        rc.setReleaseChecksumPolicy(RepositoryPolicy.CHECKSUM_POLICY_WARN);
                    } else {
                        LOGGER.warn("{} is not a supported repository checksum policy.  Please use fail, warn, ignore.", releaseChecksumPolicy);
                    }
                }
            } else {
                rc.setReleaseEnabled(false);
            }
        }

        String snapshots = props.getProperty("repository." + repositoryKeyNumber + ".snapshots.enabled", "false");
        if (ObjectUtils.isStringNotEmpty(snapshots)) {
            if (snapshots.equalsIgnoreCase("true")) {
                rc.setSnapshotEnabled(true);

                String snapshotUpdatePolicy = props.getProperty("repository." + repositoryKeyNumber + ".snapshot.policy.update", "daily");
                if (ObjectUtils.isStringNotEmpty(snapshotUpdatePolicy)) {
                    if (snapshotUpdatePolicy.equalsIgnoreCase("daily")) {
                        rc.setSnapshotUpdatePolicy(RepositoryPolicy.UPDATE_POLICY_DAILY);
                    } else if (snapshotUpdatePolicy.equalsIgnoreCase("always")) {
                        rc.setSnapshotUpdatePolicy(RepositoryPolicy.UPDATE_POLICY_ALWAYS);
                    } else if (snapshotUpdatePolicy.equalsIgnoreCase("never")) {
                        rc.setSnapshotUpdatePolicy(RepositoryPolicy.UPDATE_POLICY_NEVER);
                    } else {
                        LOGGER.warn("{} is not a supported repository update policy.  Please use always, daily or never.", snapshotUpdatePolicy);
                    }
                }

                String snapshotChecksumPolicy = props.getProperty("repository." + repositoryKeyNumber + ".snapshot.policy.checksum", "warn");
                if (ObjectUtils.isStringNotEmpty(snapshotChecksumPolicy)) {
                    if (snapshotChecksumPolicy.equalsIgnoreCase("fail")) {
                        rc.setSnapshotChecksumPolicy(RepositoryPolicy.CHECKSUM_POLICY_FAIL);
                    } else if (snapshotChecksumPolicy.equalsIgnoreCase("ignore")) {
                        rc.setSnapshotChecksumPolicy(RepositoryPolicy.CHECKSUM_POLICY_IGNORE);
                    } else if (snapshotChecksumPolicy.equalsIgnoreCase("warn")) {
                        rc.setSnapshotChecksumPolicy(RepositoryPolicy.CHECKSUM_POLICY_WARN);
                    } else {
                        LOGGER.warn("{} is not a supported repository checksum policy.  Please use fail, warn, ignore.", snapshotChecksumPolicy);
                    }
                }
            } else {
                rc.setReleaseEnabled(false);
            }
        }

        String proxyKey = "http.proxy";
        if (props.containsKey(proxyKey)) {
            URI proxyUri = new URI(props.getProperty(proxyKey));
            rc.setProxyUri(proxyUri);
        }
        return rc;
    }

    /**
     * @param propertyFileName
     * @return
     * @throws IOException
     */
    private Properties loadProperties(String propertyFileName) throws IOException {
        Properties props = null;
        try {
            ClassLoader cl = FuseCDCCLI.class.getClassLoader();
            InputStream cdcConfigStream = cl.getResourceAsStream(propertyFileName);
            props = new Properties();
            props.load(cdcConfigStream);
        } catch (Exception e) {
            LOGGER.error("Unable to load profile: " + propertyFileName);
            LOGGER.debug("  Cause: " + e.getLocalizedMessage(), e);
        }
        return props;
    }

    /**
     * Sets the String value of outputDirectory for this instance of
     * CDCTestSupport.
     * 
     * @param outputDirectory Sets String, default is TODO add default
     */
    public void setOutputDirectory(String outputDirectory) {
        this.outputDirectory = outputDirectory;
    }

    /**
     * Gets the String value of outputDirectory for this instance of
     * CDCTestSupport.
     * 
     * @return the outputDirectory
     */
    public String getOutputDirectory() {
        return outputDirectory;
    }

    /**
     * @param args
     */
    public static void main(String[] args) {

        // create the command line parser
        CommandLineParser parser = new PosixParser();

        // create Options object
        Options options = new Options();
        // add the help option
        options.addOption("h", "help", false, "Print usage information");
        // add the profile option
        options.addOption("p", "profile", true, "The name of a file that containing property overrides for "
                                                + "the default.profile. The file must be located under the etc directory.");

        String propertyFileName = null;
        try {
            // parse the command line arguments
            CommandLine line = parser.parse(options, args);
            if (line.hasOption('h')) {
                printUsage(options);
                System.exit(0);
            }
            // validate that block-size has been set
            if (line.hasOption("profile")) {
                // print the value of block-size
                // System.out.println( line.getOptionValue( "profile" ) );
                propertyFileName = line.getOptionValue("profile");
            } else {
                propertyFileName = "default.profile";
            }
        } catch (ParseException e) {
            System.out.println(e.getLocalizedMessage());
            printUsage(options);
            System.exit(100);
        }

        FuseCDCCLI cdc = new FuseCDCCLI();
        try {
            cdc.execute(propertyFileName);
        } catch (Exception e) {
            LOGGER.error("Error creating distribution: {}" + e.getLocalizedMessage(), e);
            System.exit(200);
        }
    }

    private static void printUsage(Options options) {
        HelpFormatter helpFormatter = new HelpFormatter();
        helpFormatter.setWidth(80);
        helpFormatter.printHelp(USAGE, HEADER, options, FOOTER);
    }

}
