/*
 * Copyright 2011 ArcBees Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.gwtplatform.mvp.client.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Target;

import com.gwtplatform.common.client.ProviderBundle;

/**
 * Use this annotation if you want to have certain
 * {@link com.gwtplatform.mvp.client.proxy.Proxy}s and their associated
 * {@link com.gwtplatform.mvp.client.Presenter}s to sit behind one split point
 * and to be compiled into one javascript file separately from others.
 * <p/>
 * Use this annotation if you already have too much code splitting using
 * {@link ProxyCodeSplit} and it is more efficient to group
 * {@link com.gwtplatform.mvp.client.Presenter}s because they share a bulk of
 * their code. You will also have to set up your own implementation of a
 * {@link ProviderBundle}.
 * <p/>
 * ProviderBundles can be created manually or be generated by GWTP. For the manual approach
 * the {@link ProxyCodeSplitBundle#bundleClass()} and {@link ProxyCodeSplitBundle#id()}
 * need to be defined.
 * <p/>
 * <p/>
 * Here is an example use of {@link ProxyCodeSplitBundle} using manual declaration:
 * <p/>
 * <pre>
 * &#064;ProxyCodeSplitBundle(bundleClass = MyPresenterBundle.class, id = MyPresenterBundle.ID_Object1)
 * public interface MyProxy extends ProxyPlace&lt;Object1&gt; {
 * }
 * </pre>
 * <p/>
 * <p/>
 * If you use GWTP's generation of {@link com.gwtplatform.mvp.client.ApplicationController} all bundles
 * will be automatically generated for you, all that you need are string identifiers for each unique bundle.
 * The best way to keep your bundles in order is to create an interface that identifies your bundles.
 * <pre>{@code
 * public interface Bundles {
 *   String MAIN = "Main";
 *   String OTHER = "Other";
 * }</pre>
 * <p/>
 * <p/>
 * Here is an example use of {@link ProxyCodeSplitBundle} when using GWTP's generation feature:
 * <pre>{@code
 * &#064;ProxyCodeSplitBundle(Bundles.MAIN)
 * public interface MyProxy extends ProxyPlace&lt;Object1&gt; {
 * }</pre>
 * <p/>
 *
 * @see ProviderBundle
 * @see <a href="http://code.google.com/webtoolkit/doc/latest/DevGuideCodeSplitting.html">Code Splitting</a>
 */
@Target(ElementType.TYPE)
public @interface ProxyCodeSplitBundle {

    final class NoOpProviderBundle extends ProviderBundle {
        private NoOpProviderBundle(int bundleSize) {
            super(bundleSize);
        }
    }

    Class<? extends ProviderBundle> bundleClass() default NoOpProviderBundle.class;

    int id() default -1;

    String value() default "";
}
