/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Callable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import elemental.json.Json;
import elemental.json.JsonObject;

import static com.vaadin.pro.licensechecker.Constants.LICENSE_SERVER_API_VERSION;
import static com.vaadin.pro.licensechecker.Constants.LICENSE_SERVER_LICENSES_PREFIX;
import static com.vaadin.pro.licensechecker.Util.TIMESTAMP_FORMAT;
import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Validates pre-trial licenses by communicating with the license server.
 */
class PreTrialValidator {
    private static final Logger LOGGER = LoggerFactory
            .getLogger(PreTrialValidator.class);
    private static final String VALIDATION_ERROR_MESSAGE = "Cannot validate pre-trial for {} and machineId {}. {}";
    private static final String VALIDATION_STATUS_ERROR_MESSAGE = "Cannot validate pre-trial status for {} and machineId {}. {}";
    private static final String CREATE_ERROR_MESSAGE = "Cannot create pre-trial for machineId {}";
    private static final int CONNECTION_TIMEOUT_MS = 5000;
    private static final int READ_TIMEOUT_MS = 5000;
    private static final String CONTENT_TYPE_JSON = "application/json";
    private static final String PRETRIAL_AUTH_PREFIX = "PRETRIAL ";
    private static final String VALIDATE_ENDPOINT = "/validate"
            + LICENSE_SERVER_API_VERSION + "?productName=%s&productVersion=%s";
    private static final String CREATE_ENDPOINT = "create";
    private static final String CHECK_ENDPOINT = "check";

    private final String licenseServerBaseURL;

    /**
     * Creates a new validator with the default license server URL.
     * <p>
     * A different URL for testing purposes can be provided a system property
     * using the {@code vaadin.licenseServerBaseUrl} key.
     */
    PreTrialValidator() {
        this(System.getProperty(Constants.LICENSE_SERVER_URL_PARAMETER,
                Constants.LICENSE_SERVER_URL));
    }

    /**
     * Creates a new validator with a custom license server URL.
     *
     * @param licenseServerURL
     *            the base URL of the license server
     */
    PreTrialValidator(String licenseServerURL) {
        String formattedUrl = Util.removeTrailingSlash(licenseServerURL);
        this.licenseServerBaseURL = Util.validateURL(formattedUrl);
    }

    /**
     * The possible access states returned by license validation.
     */
    public enum Access {
        /**
         * Represents a successful access state as determined by license
         * validation.
         */
        OK,
        /**
         * Indicates that access is denied as a result of license validation.
         * This state is returned when the license validation process determines
         * that the required permissions are not granted or the trial period is
         * not started or expired.
         */
        NO_ACCESS,
        /**
         * Indicates that the server could not be reached during the license
         * validation process. This state is returned when a connection error,
         * timeout, or other network-related issue occurs, preventing
         * communication with the license server. It reflects a transient or
         * connectivity problem rather than a denial of access due to licensing
         * constraints.
         */
        CANNOT_REACH_SERVER
    }

    /**
     * Base class for validation results.
     */
    public static abstract class Result {
        /**
         * Gets the access status for this result.
         *
         * @return the access status
         */
        protected abstract Access getAccess();

        /**
         * Builds an error based on the validation result.
         *
         * @param canValidateOnline
         *            whether online validation is possible
         * @return the exception, or null if no error
         */
        protected abstract LicenseValidationException buildError(
                boolean canValidateOnline);

        /**
         * Verifies access and throws an exception if needed.
         *
         * @param canValidateOnline
         *            whether online validation is possible
         * @return the access status
         * @throws LicenseValidationException
         *             if access is denied
         */
        public Access verifyAccess(boolean canValidateOnline) {
            LicenseValidationException error = buildError(canValidateOnline);
            if (error != null) {
                throw error;
            }
            return getAccess();
        }
    }

    /**
     * Result from a successful license server validation.
     */
    static final class LicenseServerResult extends Result {
        private final PreTrial preTrial;
        private final Access access;

        /**
         * Creates a new result based on trial information.
         *
         * @param preTrial
         *            the trial information
         */
        LicenseServerResult(PreTrial preTrial) {
            this.preTrial = Objects.requireNonNull(preTrial,
                    "trial cannot be null");
            this.access = preTrial.trialState == PreTrial.PreTrialState.RUNNING
                    ? Access.OK
                    : Access.NO_ACCESS;
        }

        @Override
        protected Access getAccess() {
            return access;
        }

        /**
         * Builds an appropriate {@link LicenseValidationException} based on the
         * current trial and license access state and whether online validation
         * is possible.
         * <p>
         * If the trial state is not {@code RUNNING}, an exception is always
         * returned if online validation cannot be performed. If the online
         * check is available, the method will return {@code null} ìf the trial
         * has not yet been started, in order to allow the caller to perform an
         * online validation.
         *
         * @param canValidateOnline
         *            a boolean flag indicating whether the system can perform
         *            online validation of the license.
         * @return a {@link LicenseValidationException} if validation fails
         *         under the given conditions, or {@code null} if no error is
         *         determined.
         */
        @Override
        protected LicenseValidationException buildError(
                boolean canValidateOnline) {
            if (access == Access.NO_ACCESS && !canValidateOnline) {
                return createException(preTrial);
            }
            if (canValidateOnline && access == Access.NO_ACCESS
                    && preTrial.trialState != PreTrial.PreTrialState.START_ALLOWED) {
                return createException(preTrial);
            }
            return null;
        }

        @Override
        public String toString() {
            return "Pre-trial: response from License Server [access=" + access
                    + ", " + "preTrial=" + preTrial + ']';
        }

        /*
         * When the pre-trial does not allow product usage (state is
         * ACCESS_DENIED), a basic LicenseValidationException is returned
         * instead of a more specific PreTrialLicenseValidationException. This
         * is because guiding the user to start a pre-trial that would not grant
         * access to the product would be misleading.
         */
        private static LicenseValidationException createException(
                PreTrial preTrial) {
            if (preTrial
                    .getTrialState() == PreTrial.PreTrialState.ACCESS_DENIED) {
                return new LicenseValidationException(
                        "Pre trial does not grant access to the product.");
            }
            return new PreTrialLicenseValidationException(preTrial);
        }

    }

    /**
     * Result from a failed license server request.
     */
    static final class RequestFailure extends Result {
        private final Access access;
        private final String errorMessage;

        /**
         * Creates a new result for a failed request.
         *
         * @param access
         *            the access status
         * @param errorMessage
         *            the error message
         */
        RequestFailure(Access access, String errorMessage) {
            this.access = access;
            this.errorMessage = errorMessage;
        }

        @Override
        protected Access getAccess() {
            return access;
        }

        @Override
        protected LicenseValidationException buildError(
                boolean canValidateOnline) {
            if (access == Access.CANNOT_REACH_SERVER && canValidateOnline) {
                return null;
            }
            return new LicenseValidationException(errorMessage);
        }

        @Override
        public String toString() {
            return "Pre-Trial: request to License Server failed [access="
                    + access + ", errorMessage=" + errorMessage + ']';
        }

        /**
         * Gets the error message from this failure.
         *
         * @return the error message, may be null
         */
        String getErrorMessage() {
            return errorMessage;
        }
    }

    static final class RecentlyValidated extends Result {

        @Override
        protected Access getAccess() {
            return Access.OK;
        }

        @Override
        protected LicenseValidationException buildError(
                boolean canValidateOnline) {
            return null;
        }

        @Override
        public String toString() {
            return "Pre-trial: recently validated [access=OK]";
        }
    }

    /**
     * Requests the creation of a new pre-trial period for the given machine
     * identifier and user key.
     * <p>
     * If a pre-trial is already started it returns the state with updated
     * remaining days. Otherwise, if the pre-trial has expired but cannot yet be
     * renewed a {@link PreTrialCreationException.Expired} is thrown.
     *
     * @param machineId
     *            the machine ID
     * @param userKey
     *            the user key
     * @return detail about the pre-trial.
     * @throws PreTrialCreationException.Expired
     *             if the pre-trial is expired and cannot yet be renewed
     * @throws PreTrialCreationException
     *             if the License Server prevents the start of a pre-trial
     */
    PreTrial create(String machineId, String userKey) {
        validateStringParameter("machineId", machineId);
        validateStringParameter("userKey", userKey);
        LOGGER.trace(
                "Submitting PRE TRIAL create request for machineId {} and userKey {}",
                machineId, userKey);
        Callable<HttpURLConnection> urlConnectionFactory = () -> getHttpURLConnection(
                licenseServerBaseURL + Constants.LICENSE_SERVER_PRETRIAL_URL
                        + CREATE_ENDPOINT,
                "POST", machineId);
        try {
            JsonObject requestBody = Json.createObject();
            requestBody.put("machineId", machineId);
            requestBody.put("userKey", userKey);
            requestBody.put("timestamp",
                    TIMESTAMP_FORMAT.format(Instant.now()));

            Response response = submitRequest(urlConnectionFactory,
                    requestBody);
            Result result = parseResponse(response, false);
            LOGGER.debug("Pre-trial create result: {}", result);
            if (result instanceof LicenseServerResult) {
                return ((LicenseServerResult) result).preTrial;
            }
            RequestFailure failure = (RequestFailure) result;
            if (response.status == 401) {
                throw new PreTrialCreationException.Expired(
                        failure.getErrorMessage());
            }
            throw new PreTrialCreationException(failure.getErrorMessage());
        } catch (PreTrialCreationException e) {
            throw e;
        } catch (Exception e) {
            LOGGER.error(CREATE_ERROR_MESSAGE, machineId, e);
            throw new PreTrialCreationException(e.getMessage());
        }
    }

    /**
     * Validates if a trial is allowed for the given product.
     *
     * @param product
     *            the product to validate
     * @param machineId
     *            the machine ID
     * @param quiet
     *            whether to log errors as debug instead of error
     * @return the validation result
     */
    Result validate(Product product, String machineId, boolean quiet) {
        validateStringParameter("machineId", machineId);
        validateProduct(product);
        if (History.isPreTrialRecentlyValidated(product, machineId)) {
            LOGGER.debug(
                    "Skipping check as pre-trial license was recently validated and not yet expired.");
            return new RecentlyValidated();
        }
        try {
            LOGGER.trace(
                    "Submitting PRE TRIAL validate request for machineId {} and {}",
                    machineId, product);
            Callable<HttpURLConnection> urlConnectionFactory = () -> getHttpURLConnection(
                    licenseServerBaseURL + LICENSE_SERVER_LICENSES_PREFIX
                            + VALIDATE_ENDPOINT,
                    "GET", machineId, product.getName(), product.getVersion());
            Result result = parseResponse(
                    submitRequest(urlConnectionFactory, null), true);
            LOGGER.debug("Validation result: {}", result);
            if (result.getAccess() == Access.OK) {
                History.setLastPreTrialCheckTime(product, machineId,
                        Instant.now(), ((LicenseServerResult) result).preTrial);
            }
            logResponseErrors(result, product, machineId, quiet);
            return result;
        } catch (Exception e) {
            if (quiet) {
                LOGGER.debug(VALIDATION_ERROR_MESSAGE, product, machineId,
                        e.getMessage());
            } else {
                LOGGER.error(VALIDATION_ERROR_MESSAGE, product, machineId, "",
                        e);
            }
        }

        return new RequestFailure(Access.CANNOT_REACH_SERVER, null);
    }

    /**
     * Gets the status of a pre-trial period for the given product and machine
     * identifier.
     * <p>
     * It returns the state of the pre-trial with information about remaining
     * days.
     *
     * @param product
     *            the product to check
     * @param machineId
     *            the machine ID
     * @return the pre-trial status
     * @throws PreTrialStatusCheckException
     *             if pre-trial status cannot be checked against the License
     *             Server.
     */
    Optional<PreTrial> statusCheck(Product product, String machineId) {
        validateStringParameter("machineId", machineId);
        validateProduct(product);
        PreTrial lastKnownPreTrial = History
                .getRecentlyValidatedPreTrial(product, machineId);
        if (lastKnownPreTrial != null) {
            return Optional.of(lastKnownPreTrial);
        }
        try {
            LOGGER.trace(
                    "Submitting PRE TRIAL validate request for status check for machineId {} and {}",
                    machineId, product);
            Callable<HttpURLConnection> urlConnectionFactory = () -> getHttpURLConnection(
                    licenseServerBaseURL + LICENSE_SERVER_LICENSES_PREFIX
                            + VALIDATE_ENDPOINT,
                    "GET", machineId, product.getName(), product.getVersion());
            Result result = parseResponse(
                    submitRequest(urlConnectionFactory, null), true);
            LOGGER.debug("Validation result: {}", result);
            logResponseErrors(result, product, machineId, true,
                    VALIDATION_STATUS_ERROR_MESSAGE);
            if (result.getAccess() == Access.OK) {
                History.setLastPreTrialCheckTime(product, machineId,
                        Instant.now(), ((LicenseServerResult) result).preTrial);
            }
            if (result instanceof RequestFailure) {
                throw new PreTrialStatusCheckException(
                        ((RequestFailure) result).getErrorMessage(), result.getAccess() == Access.CANNOT_REACH_SERVER);
            }
            return result instanceof LicenseServerResult
                    ? Optional.of(((LicenseServerResult) result).preTrial)
                    : Optional.empty();
        } catch (Exception e) {
            LOGGER.error(VALIDATION_STATUS_ERROR_MESSAGE, product, machineId,
                    "", e);
            throw new PreTrialStatusCheckException(e.getMessage(), true);
        }
    }

    /**
     * Gets the status of a pre-trial period for the given product, machine
     * identifier and user key.
     * <p>
     * It returns the state of the active pre-trial with information about
     * remaining days, or an empty {@link Optional} if the pre-trial has not yet
     * been started. If the pre-trial has expired but cannot yet be renewed a
     * {@link PreTrialCheckException.Expired} is thrown.
     *
     * @param product
     *            the product to validate
     * @param machineId
     *            the machine ID
     * @param userKey
     *            the user key
     * @return detail about the pre-trial.
     * @throws PreTrialCheckException.Expired
     *             if the pre-trial is expired and cannot yet be renewed
     * @throws PreTrialCheckException
     *             if the product is not eligible for a pre-trial, or if
     *             pre-trial status cannot be checked against the License
     *             Server.
     */
    Optional<PreTrial> check(Product product, String machineId,
            String userKey) {
        validateStringParameter("machineId", machineId);
        validateStringParameter("userKey", userKey);
        validateProduct(product);
        LOGGER.trace(
                "Submitting PRE TRIAL check request for machineId {}, userKey {} and {}",
                machineId, userKey, product);
        Callable<HttpURLConnection> urlConnectionFactory = () -> getHttpURLConnection(
                licenseServerBaseURL + Constants.LICENSE_SERVER_PRETRIAL_URL
                        + CHECK_ENDPOINT,
                "POST", machineId);
        try {
            JsonObject requestBody = Json.createObject();
            requestBody.put("machineId", machineId);
            requestBody.put("userKey", userKey);
            requestBody.put("productName", product.getName());
            requestBody.put("productVersion", product.getVersion());
            requestBody.put("timestamp",
                    TIMESTAMP_FORMAT.format(Instant.now()));

            Response response = submitRequest(urlConnectionFactory,
                    requestBody);
            Result result = parseResponse(response, false);
            LOGGER.debug("Pre-trial create result: {}", result);
            if (result instanceof LicenseServerResult) {
                return Optional.of(((LicenseServerResult) result).preTrial);
            }
            RequestFailure failure = (RequestFailure) result;
            if (response.status == 404) {
                return Optional.empty();
            } else if (response.status == 401) {
                throw new PreTrialCheckException.Expired(
                        failure.getErrorMessage());
            }
            throw new PreTrialCheckException(failure.getErrorMessage());
        } catch (PreTrialCheckException e) {
            throw e;
        } catch (Exception e) {
            LOGGER.error(CREATE_ERROR_MESSAGE, machineId, e);
            throw new PreTrialCheckException(e.getMessage());
        }
    }

    private static void validateProduct(Product product) {
        if (product == null) {
            throw new IllegalArgumentException("product cannot be null");
        }
    }

    private static void validateStringParameter(String paramName,
            String paramValue) {
        if (paramValue == null || paramValue.trim().isEmpty()) {
            throw new IllegalArgumentException(
                    paramName + " cannot be null or empty");
        }
    }

    private void logResponseErrors(Result result, Product product,
            String machineId, boolean quiet) {
        logResponseErrors(result, product, machineId, quiet,
                VALIDATION_ERROR_MESSAGE);
    }

    private void logResponseErrors(Result result, Product product,
            String machineId, boolean quiet, String messagePattern) {
        if (result instanceof RequestFailure) {
            String message = ((RequestFailure) result).getErrorMessage();
            if (message != null) {
                if (quiet) {
                    LOGGER.debug(messagePattern, product, machineId, message);
                } else {
                    LOGGER.error(messagePattern, product, machineId, "");
                }
            }
        }
    }

    private Result parseResponse(Response response, boolean validate)
            throws IOException {

        JsonObject body = response.body;
        LOGGER.trace("License Server response: {}", body.toJson());

        if (body.hasKey("error") && body.hasKey("message")) {
            return createErrorResult(body);
        } else if (body.hasKey("trialName")) {
            return createTrialResult(body, validate);
        }

        throw new IOException("Invalid response from License Server: " + body);
    }

    private Result createTrialResult(JsonObject response, boolean validate)
            throws IOException {
        PreTrial.PreTrialState trialState;
        if (validate) {
            try {
                trialState = PreTrial.PreTrialState
                        .valueOf(response.getString("trialState"));
            } catch (IllegalArgumentException e) {
                throw new IOException(
                        "Invalid trial state from License Server: " + response);
            }
        } else {
            trialState = PreTrial.PreTrialState.RUNNING;
        }

        PreTrial preTrial = new PreTrial(response.getString("trialName"),
                trialState, (int) response.getNumber("daysRemaining"),
                (int) response.getNumber("daysRemainingUntilRenewal"));

        return new LicenseServerResult(preTrial);
    }

    private Result createErrorResult(JsonObject response) {
        StringBuilder errorMessageBuilder = new StringBuilder("Error [")
                .append(response.getString("error")).append("] ")
                .append(response.getString("message"));

        if (response.hasKey("detail")) {
            errorMessageBuilder.append(". ")
                    .append(response.getString("detail"));
        }

        if (response.hasKey("path")) {
            errorMessageBuilder.append(". (Path: ")
                    .append(response.getString("path")).append(")");
        }

        return new RequestFailure(Access.NO_ACCESS,
                errorMessageBuilder.toString());
    }

    private Response submitRequest(
            Callable<HttpURLConnection> urlConnectionFactory,
            JsonObject requestBody) throws Exception {

        HttpURLConnection http = urlConnectionFactory.call();
        if (requestBody != null) {
            try (OutputStream os = http.getOutputStream()) {
                os.write(requestBody.toJson().getBytes(UTF_8));
            }
        }

        int status = http.getResponseCode();

        if (status >= 200 && status < 300) {
            return new Response(status,
                    Util.parseJson(status, http.getInputStream()));
        } else if (status >= 400 && status < 600) {
            return new Response(status,
                    Util.parseJson(status, http.getErrorStream()));
        }

        throw new IOException(
                "Unexpected response from License Server: " + status);
    }

    private static class Response {
        private final int status;
        private final JsonObject body;

        Response(int status, JsonObject body) {
            this.status = status;
            this.body = body;
        }
    }

    private HttpURLConnection getHttpURLConnection(String endpointURL,
            String httpMethod, String machineId, String... queryParams)
            throws IOException {
        URL url = new URL(String.format(endpointURL, (Object[]) queryParams));

        HttpURLConnection http = (HttpURLConnection) url.openConnection();

        http.setRequestMethod(httpMethod);
        http.setRequestProperty("Content-Type", CONTENT_TYPE_JSON);
        http.setRequestProperty("Accept", CONTENT_TYPE_JSON);
        http.setRequestProperty("Authorization",
                PRETRIAL_AUTH_PREFIX + machineId);

        http.setConnectTimeout(CONNECTION_TIMEOUT_MS);
        http.setReadTimeout(READ_TIMEOUT_MS);
        http.setDoOutput(true);
        return http;
    }

}
