/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.flow.component.shared.internal;

import java.io.Serializable;
import java.util.Objects;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ModalityMode;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.dom.DisabledUpdateMode;
import com.vaadin.flow.function.SerializableSupplier;
import com.vaadin.flow.internal.StateTree;
import com.vaadin.flow.shared.Registration;

/**
 * An internal controller for automatically adding a component to the UI when
 * it's opened. Not intended to be used publicly.
 *
 * @param <C>
 *            Type of the component that uses this controller.
 */
public class OverlayAutoAddController<C extends Component>
        implements Serializable {
    private final C component;
    private final SerializableSupplier<ModalityMode> modalityModeSupplier;

    private boolean skipOnNavigation;
    private boolean autoAdded;
    private Registration beforeLeaveRegistration;

    public OverlayAutoAddController(C component) {
        this(component, () -> ModalityMode.MODELESS);
    }

    public OverlayAutoAddController(C component,
            SerializableSupplier<ModalityMode> modalityModeSupplier) {
        this.component = component;
        this.modalityModeSupplier = Objects.requireNonNull(modalityModeSupplier,
                "modalityModeSupplier must not be null");

        // Automatically add the component to the UI when it is opened.
        component.getElement().addPropertyChangeListener("opened", event -> {
            if (isOpened()) {
                handleOpen();
            }
        });

        // Automatically remove the component from the UI after the overlay's
        // closing animation has finished. This way auto-removal works by first
        // setting `opened` on the client-side to `false` and then waiting for
        // the `closed` event from the client-side.
        // The event needs to be allowed for inert components so that closing
        // from the server still works. This requires double-checking that the
        // component is actually in a closed state on the server.
        // Also allow the event on disabled components, as LoginOverlay for
        // example disables itself on the login event.
        component.getElement().addEventListener("closed", event -> {
            if (!isOpened()) {
                handleClose();
            }
        }).allowInert().setDisabledUpdateMode(DisabledUpdateMode.ALWAYS);
    }

    /**
     * Adds the component to the UI if it is not already attached, regardless of
     * the current opened state of the component. Allows components with custom
     * opening logic, such as ContextMenu, to add themselves to the UI and reuse
     * the automatic removal logic when the component is closed.
     */
    public void add() {
        if (!isAttached()) {
            UI ui = getUI();
            ui.addToModalComponent(component);
            ui.setChildComponentModal(component, modalityModeSupplier.get());
            autoAdded = true;
        }
    }

    /**
     * Force remove the component from the UI in case it was auto-added. Can be
     * used by components with custom closing logic. For example, Notification
     * removes itself from the UI whenever it is detached.
     */
    public void remove() {
        handleClose();
    }

    /**
     * Sets whether to skip auto-adding when the UI navigates to a new view
     * before the component is opened.
     *
     * @param skipOnNavigation
     *            whether to skip auto-adding on navigation
     */
    public void setSkipOnNavigation(boolean skipOnNavigation) {
        this.skipOnNavigation = skipOnNavigation;
    }

    private void handleOpen() {
        UI ui = getUI();
        StateTree.ExecutionRegistration addToUiRegistration = ui
                .beforeClientResponse(ui, context -> {
                    if (isOpened()) {
                        add();
                    }
                    if (beforeLeaveRegistration != null) {
                        beforeLeaveRegistration.remove();
                        beforeLeaveRegistration = null;
                    }
                });

        if (skipOnNavigation && ui.getSession() != null) {
            // Cancel auto-adding if the current view is navigated away from
            // before the component is added to the UI.
            beforeLeaveRegistration = ui.addBeforeLeaveListener(event -> {
                addToUiRegistration.remove();
                beforeLeaveRegistration.remove();
                beforeLeaveRegistration = null;
            });
        }
    }

    private void handleClose() {
        if (autoAdded) {
            autoAdded = false;
            component.getElement().removeFromParent();
        }
    }

    private UI getUI() {
        UI ui = UI.getCurrent();
        if (ui == null) {
            throw new IllegalStateException("UI instance is not available. "
                    + "It means that you are calling this method "
                    + "out of a normal workflow where it's always implicitly set. "
                    + "That may happen if you call the method from the custom thread without "
                    + "'UI::access' or from tests without proper initialization.");
        }
        return ui;
    }

    private boolean isOpened() {
        return component.getElement().getProperty("opened", false);
    }

    private boolean isAttached() {
        return component.getElement().getNode().getParent() != null;
    }
}
