/**
 *  Copyright 2005-2015 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package io.fabric8.patch.management;

import java.io.File;
import java.net.ProtocolException;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.util.List;
import java.util.Map;

import org.eclipse.jgit.api.Git;

/**
 * <p>Interface to an OSGi service that can handle low-level patch management operations.</p>
 * <p>To be used by patch:* commands when interaction with patch data is required.</p>
 */
public interface PatchManagement {

    /**
     * Load all available patches. {@link Patch#getManagedPatch()} returns only basic information.
     * To get more details, use {@link #loadPatch(PatchDetailsRequest)} with specific details.
     * @param details whether to retrieve {@link ManagedPatch} information
     * @return
     * @throws PatchException
     */
    List<Patch> listPatches(boolean details) throws PatchException;

    /**
     * Load {@link Patch} with required details level, specified in {@link PatchDetailsRequest}.
     * @param request
     * @return
     */
    Patch loadPatch(PatchDetailsRequest request) throws PatchException;

    /**
     * Retriees an artifact from a given URL and returns list of patch data it contains (*.zip) or it is (*.patch)
     * @param url
     * @return
     */
    List<PatchData> fetchPatches(URL url) throws PatchException;

    /**
     * Artifacts referenced from patch are uploaded to <code>URI</code>
     * @param patchData
     * @param uploadAddress
     * @param callback
     * @throws PatchException
     */
    void uploadPatchArtifacts(PatchData patchData, URI uploadAddress, UploadCallback callback) throws PatchException;

    /**
     * Takes already downloaded {@link PatchData static patch data} and prepares the patch to be installed,
     * examined and/or simulated
     * @param patchData
     * @return
     */
    Patch trackPatch(PatchData patchData) throws PatchException;

    /**
     * <p>We can install many patches at once, but those have to be of the same {@link PatchKind}.</p>
     * <p>This method returns a <em>handle</em> to <em>patch transaction</em> that has to be passed around
     * when installing consecutive patches. This <em>handle</em> is effectively a temporary branch name when patches
     * are applied</p>
     * @param kind
     * @return
     */
    String beginInstallation(PatchKind kind);

    /**
     * <p>When patch installation <em>transaction</em> is started, we can install as many {@link PatchKind#NON_ROLLUP}
     * patches we like or install single {@link PatchKind#ROLLUP} patch</p>
     * <p>This won't affect ${karaf.home} until we {@link #commitInstallation(String) commit the installation}.</p>
     * @param transaction
     * @param patch
     * @param bundleUpdatesInThisPatch
     */
    void install(String transaction, Patch patch, List<BundleUpdate> bundleUpdatesInThisPatch);

    /**
     * <p>After successful patch(es) installation, we commit the transaction = do a fast forward merge
     * of main patch branch to transaction branch</p>
     * @param transaction
     */
    void commitInstallation(String transaction);

    /**
     * <p>When something goes wrong (or during simulation?) we delete transaction branch without affecting
     * main patch branch.</p>
     * @param transaction
     */
    void rollbackInstallation(String transaction);

    /**
     * <p>Rolling back a patch (which is different that rolling back transaction for patches installation) is
     * a matter of restoring to previous point in history</p>
     * @param patchData
     */
    void rollback(PatchData patchData);

    /**
     * Fabric-mode operation - takes external {@link Git} and updates a branch (version) with new profiles
     * shipped with new {@link PatchKind#ROLLUP rollup patch}.
     * @param gitRepository
     * @param versionId
     * @param patch
     * @param strategy
     */
    void installProfiles(File gitRepository, String versionId, Patch patch, ProfileUpdateStrategy strategy);

    /**
     * Patching services can be called from high level services. This method takes a map of versions
     * (<code>io.fabric8.version</code> PID). It's job is to update static resources of the container.
     * This method should be called in fabric mode.
     * @param versions
     * @param urls list of urls of critical bundles that have to be downloaded to ${karaf.default.repository}
     * @param localMavenRepository
     * @param callback
     * @return <code>true</code> if container needs restart
     */
    boolean alignTo(Map<String, String> versions, List<String> urls, File localMavenRepository, Runnable callback) throws PatchException;

    /**
     * Checks whether we're in <code>admin:create</code> based child container
     * @return
     */
    boolean isStandaloneChild();

    /**
     * Callback to be passed to method that uploads content of retrieved patches to remote repository.
     */
    interface UploadCallback {

        void doWithUrlConnection(URLConnection connection) throws ProtocolException;

    }

}
