package com.logviewer.logLibs.logback;

import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.encoder.PatternLayoutEncoder;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import ch.qos.logback.core.FileAppender;
import ch.qos.logback.core.encoder.Encoder;
import ch.qos.logback.core.pattern.parser.Node;
import ch.qos.logback.core.pattern.parser.Parser;
import ch.qos.logback.core.spi.AppenderAttachable;
import ch.qos.logback.core.spi.ScanException;
import com.logviewer.data2.LogFormat;
import com.logviewer.data2.config.VisibleDirectory;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.nio.file.Path;
import java.util.*;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class LogbackConfigImporter implements Supplier<Map<Path, LogFormat>> {

    private static final org.slf4j.Logger LOG = LoggerFactory.getLogger(LogbackConfigImporter.class);

    private static void processAppender(Appender<ILoggingEvent> appender, Set<Appender<ILoggingEvent>> processedAppenders,
                                        ArrayList<VisibleDirectory> visibleDirectories, Map<Path, LogFormat> res) {
        if (!processedAppenders.add(appender))
            return;

        if (appender instanceof FileAppender<?>) {
            FileAppender fileAppender = (FileAppender) appender;

            Encoder encoder = fileAppender.getEncoder();

            if (!(encoder instanceof PatternLayoutEncoder))
                throw new IllegalStateException("Failed to import log config - unsupported encoder: " + encoder.getClass());

            try {
                File file = new File(fileAppender.getFile()).getCanonicalFile();

                File parent = file.getParentFile();
                if (parent == null)
                    return;

                PatternLayoutEncoder patternEncoder = (PatternLayoutEncoder) encoder;

                String pattern = patchPatternWithProcessId(patternEncoder.getPattern());

                LogFormat logFormat = new LogbackLogFormat(pattern).setCharset(patternEncoder.getCharset());

                try {
                    logFormat.validate();
                } catch (IllegalArgumentException e) {
                    LOG.error("Failed to import log configuration, invalid pattern: " + patternEncoder.getPattern(), e);
                    logFormat = null;
                }

                visibleDirectories.add(new VisibleDirectory(parent.getPath(), Pattern.quote(file.getName())));

                res.put(file.toPath(), logFormat);
            } catch (IOException e) {
                LOG.error("Failed to import log configuration", e);
            }
        } else if (appender instanceof AppenderAttachable) {
            Iterator<Appender<ILoggingEvent>> itr = ((AppenderAttachable) appender).iteratorForAppenders();
            while (itr.hasNext()) {
                processAppender(itr.next(), processedAppenders, visibleDirectories, res);
            }
        }
    }

    protected static String getProcessId() {
        String name = ManagementFactory.getRuntimeMXBean().getName();
        Matcher matcher = Pattern.compile("(\\d+)@").matcher(name);
        if (!matcher.lookingAt())
            return null;

        return matcher.group(1);
    }

    /**
     * String-boot may insert PID into pattern as a plain string. We have to replace it with `%processId` to support
     * logs generated by previous application instance.
     */
    protected static String patchPatternWithProcessId(String pattern) {
        String processId = getProcessId();
        if (processId == null)
            return pattern;

        Pattern pidPattern = Pattern.compile("\\b" + processId + "\\b");

        Matcher matcher = pidPattern.matcher(pattern);
        if (!matcher.find())
            return pattern;

        if (matcher.find())
            return pattern; // Only one PID expected

        if (!hasLiteral(pattern, pidPattern))
            return pattern;

        return pidPattern.matcher(pattern).replaceFirst("%processId");
    }

    private static boolean hasLiteral(String pattern, Pattern pid) {
        try {
            Parser parser = new Parser(pattern);
            Node node = parser.parse();

            while (node != null) {
                if (node.getType() == LogbackLogFormat.NODE_LITERAL) {
                    Matcher matcher = pid.matcher((String) node.getValue());
                    if (matcher.find())
                        return true;
                }
                node = node.getNext();
            }

            return false;
        } catch (ScanException e) {
            return false;
        }
    }

    @Override
    public Map<Path, LogFormat> get() {
        LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();

        ArrayList<VisibleDirectory> visibleDirectories = new ArrayList<>();

        Set<Appender<ILoggingEvent>> processedAppenders = new HashSet<>();

        Map<Path, LogFormat> res = new HashMap<>();

        for (Logger logger : loggerContext.getLoggerList()) {
            for (Iterator<Appender<ILoggingEvent>> index = logger.iteratorForAppenders(); index.hasNext();) {
                processAppender(index.next(), processedAppenders, visibleDirectories, res);
            }
        }

        return res;
    }
}
