/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.oozie.fluentjob.api.workflow;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.oozie.fluentjob.api.ModifyOnce;
import org.apache.oozie.fluentjob.api.action.Builder;
import org.apache.oozie.fluentjob.api.action.Node;

import java.security.SecureRandom;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Queue;
import java.util.Set;

/**
 * A builder class for {@link Workflow}.
 * <p>
 * The properties of the builder can only be set once, an attempt to set them a second time will trigger
 * an {@link IllegalStateException}. The properties that are lists are an exception to this rule, of course multiple
 * elements can be added / removed.
 * <p>
 * Builder instances can be used to build several elements, although properties already set cannot be changed after
 * a call to {@link WorkflowBuilder#build} either.
 * <p>
 * Being the topmost {@link Builder} from the user's perspective, it serves as a main entry point to create
 * Workflow API POJOs.
 */
@InterfaceAudience.Private
@InterfaceStability.Unstable
public class WorkflowBuilder implements Builder<Workflow> {
    private final ModifyOnce<String> name;
    private final List<Node> addedActions;
    private ParametersBuilder parametersBuilder;
    private GlobalBuilder globalBuilder;
    private CredentialsBuilder credentialsBuilder;

    public WorkflowBuilder() {
        this.name = new ModifyOnce<>();
        this.addedActions = new ArrayList<>();
    }

    public WorkflowBuilder withName(final String name) {
        this.name.set(name);
        return this;
    }

    public WorkflowBuilder withDagContainingNode(final Node node) {
        this.addedActions.add(node);
        return this;
    }

    public WorkflowBuilder withParameter(final String name, final String value) {
        ensureParametersBuilder();
        this.parametersBuilder.withParameter(name, value);
        return this;
    }

    public WorkflowBuilder withParameter(final String name, final String value, final String description) {
        this.parametersBuilder.withParameter(name, value, description);
        return this;
    }

    private void ensureParametersBuilder() {
        if (this.parametersBuilder == null) {
            this.parametersBuilder = ParametersBuilder.create();
        }
    }

    public WorkflowBuilder withGlobal(final Global global) {
        this.globalBuilder = GlobalBuilder.createFromExisting(global);
        return this;
    }

    /**
     * Adds a {@link Credentials} to the {@link Workflow}.
     * <p>
     * Note that an explicit call to {@code WorkflowBuilder#withCredentials(Credentials)} can be omitted, since if
     * {@link WorkflowBuilder#credentialsBuilder} is emtpy by the time {@link WorkflowBuilder#build()} is called,
     * {@link Workflow#credentials} is built based on all the {@link Node#getCredentials()} that have been added to that
     * {@code Workflow} in beforehand.
     * <p>
     * Note also that when {@code WorkflowBuilder#withCredentials(Credentials)} is explicitly called, the {@code <workflowapp />}'s
     * {@code <credential />} section is generated only by using the {@code Credentials} defined on the {@code Workflow} level.
     * <p>
     * This way, users can, if they want to, omit calling {@code WorkflowBuilder#withCredentials(Credentials)} by default, but can
     * also override the autogenerated {@code <credentials />} section of {@code <workflowapp />} by explicitly calling that method
     * after another call to {@link CredentialsBuilder#build()}.
     * @param credentials the {@code Credentials} to add, if want to override.
     * @return this
     */
    public WorkflowBuilder withCredentials(final Credentials credentials) {
        this.credentialsBuilder = CredentialsBuilder.createFromExisting(credentials);
        return this;
    }

    @Override
    public Workflow build() {
        ensureName();

        final Set<Node> nodes = new HashSet<>();
        for (final Node node : addedActions) {
            if (!nodes.contains(node)) {
                nodes.addAll(getNodesInDag(node));
            }
        }

        final ImmutableSet.Builder<Node> builder = new ImmutableSet.Builder<>();
        builder.addAll(nodes);

        final Parameters parameters;
        if (parametersBuilder != null) {
            parameters = parametersBuilder.build();
        }
        else {
            parameters = null;
        }

        final Global global;
        if (globalBuilder != null) {
            global = globalBuilder.build();
        }
        else {
            global = null;
        }

        final Credentials credentials;
        if (credentialsBuilder != null) {
            credentials = credentialsBuilder.build();
        }
        else {
            final CredentialsBuilder actionCredentialsBuilder = CredentialsBuilder.create();
            for (final Node action : addedActions) {
                for (final Credential actionCredential : action.getCredentials()) {
                    actionCredentialsBuilder.withCredential(actionCredential);
                }
            }

            final Credentials actionCredentials = actionCredentialsBuilder.build();
            if (actionCredentials.getCredentials().size() > 0) {
                credentials = actionCredentialsBuilder.build();
            } else {
                credentials = null;
            }
        }

        return new Workflow(name.get(), builder.build(), parameters, global, credentials);
    }

    private void ensureName() {
        if (Strings.isNullOrEmpty(this.name.get())) {
            final String type = "workflow";
            final int randomSuffix = new SecureRandom().nextInt(1_000_000_000);

            this.name.set(String.format("%s-%d", type, randomSuffix));
        }
    }

    private static Set<Node> getNodesInDag(final Node node) {
        final Set<Node> visited = new HashSet<>();
        final Queue<Node> queue = new ArrayDeque<>();
        visited.add(node);
        queue.add(node);

        Node current;
        while ((current = queue.poll()) != null) {
            visit(current.getAllParents(), visited, queue);
            visit(current.getAllChildren(), visited, queue);
        }

        return visited;
    }

    // TODO: encapsulate into a more specific nested class, e.g. DagWalker#walk
    private static void visit(final List<Node> toVisit, final Set<Node> visited, final Queue<Node> queue) {
        for (final Node node : toVisit) {
            if (!visited.contains(node)) {
                visited.add(node);
                queue.add(node);
            }
        }
    }
}
