/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.servicemix.executors.impl;

/**
 * This bean holds configuration attributes for a given Executor.
 * 
 * @author <a href="mailto:gnodet [at] gmail.com">Guillaume Nodet</a>
 */
public class ExecutorConfig {

    public static final int DEFAULT_CORE_POOL_SIZE = 4;
    public static final int DEFAULT_MAXIMUM_POOL_SIZE = -1;
    public static final long DEFAULT_KEEP_ALIVE_TIME = 60000;
    public static final boolean DEFAULT_THREAD_DAEMON = false;
    public static final int DEFAULT_THREAD_PRIORITY = Thread.NORM_PRIORITY;
    public static final int DEFAULT_QUEUE_SIZE = 1024;
    public static final long DEFAULT_SHUTDOWN_DELAY = 1000;
    public static final boolean DEFAULT_ALLOW_CORE_THREAD_TIMEOUT = true;
    public static final boolean DEFAULT_BYPASS_IF_SYNCHRONOUS = false;

    private ExecutorConfig parent;

    private Integer corePoolSize;

    private Integer maximumPoolSize;

    private Long keepAliveTime;

    private Boolean threadDaemon;

    private Integer threadPriority = Thread.NORM_PRIORITY;

    private Integer queueSize;

    private Long shutdownDelay;

    private Boolean allowCoreThreadsTimeout;

    private Boolean bypassIfSynchronous;

    /**
     * default constructor needed by spring beans
     */
    public ExecutorConfig() {
        this(true, null);
    }

    /**
     * creates a new executor config using the given parent
     *
     * @param parent the parent config
     */
    public ExecutorConfig(boolean isDefaultConfig, ExecutorConfig parent) {
        this.parent = parent;
        // if this is the default config we don't want undefined values
        if (isDefaultConfig) {
            setQueueSize(DEFAULT_QUEUE_SIZE);
            setShutdownDelay(DEFAULT_SHUTDOWN_DELAY);
            setThreadDaemon(DEFAULT_THREAD_DAEMON);
            setThreadPriority(DEFAULT_THREAD_PRIORITY);
            setAllowCoreThreadsTimeout(DEFAULT_ALLOW_CORE_THREAD_TIMEOUT);
            setBypassIfSynchronous(DEFAULT_BYPASS_IF_SYNCHRONOUS);
            setCorePoolSize(DEFAULT_CORE_POOL_SIZE);
            setKeepAliveTime(DEFAULT_KEEP_ALIVE_TIME);
            setMaximumPoolSize(DEFAULT_MAXIMUM_POOL_SIZE);
        }
    }

    /**
     * @return the corePoolSize
     */
    public int getCorePoolSize() {
        return getParent() != null && corePoolSize == null ? getParent().getCorePoolSize() : corePoolSize;
    }

    /**
     * @param corePoolSize
     *            the corePoolSize to set
     */
    public void setCorePoolSize(int corePoolSize) {
        this.corePoolSize = corePoolSize;
    }

    /**
     * @return the keepAlive
     */
    public long getKeepAliveTime() {
        return getParent() != null && keepAliveTime == null ? getParent().getKeepAliveTime() : keepAliveTime;
    }

    /**
     * @param keepAlive
     *            the keepAlive to set
     */
    public void setKeepAliveTime(long keepAlive) {
        this.keepAliveTime = keepAlive;
    }

    /**
     * @return the maximumPoolSize
     */
    public int getMaximumPoolSize() {
        return getParent() != null && maximumPoolSize == null ? getParent().getMaximumPoolSize() : maximumPoolSize;
    }

    /**
     * @param maximumPoolSize
     *            the maximumPoolSize to set
     */
    public void setMaximumPoolSize(int maximumPoolSize) {
        this.maximumPoolSize = maximumPoolSize;
    }

    /**
     * @return the queueSize
     */
    public int getQueueSize() {
        return getParent() != null && queueSize == null ? getParent().getQueueSize() : queueSize;
    }

    /**
     * @param queueSize
     *            the queueSize to set
     */
    public void setQueueSize(int queueSize) {
        this.queueSize = queueSize;
    }

    /**
     * @return the threadDaemon
     */
    public boolean isThreadDaemon() {
        return getParent() != null && threadDaemon == null ? getParent().isThreadDaemon() : threadDaemon;
    }

    /**
     * @param threadDaemon
     *            the threadDaemon to set
     */
    public void setThreadDaemon(boolean threadDaemon) {
        this.threadDaemon = threadDaemon;
    }

    /**
     * @return the threadPriority
     */
    public int getThreadPriority() {
        return getParent() != null && threadPriority == null ? getParent().getThreadPriority() : threadPriority;
    }

    /**
     * @param threadPriority
     *            the threadPriority to set
     */
    public void setThreadPriority(int threadPriority) {
        this.threadPriority = threadPriority;
    }

    /**
     * @return the shutdownDelay
     */
    public long getShutdownDelay() {
        return getParent() != null && shutdownDelay == null ? getParent().getShutdownDelay() : shutdownDelay;
    }

    /**
     * @param shutdownDelay
     *            the shutdownDelay to set
     */
    public void setShutdownDelay(long shutdownDelay) {
        this.shutdownDelay = shutdownDelay;
    }

    /**
     * @return the allowCoreThreadsTimeout
     */
    public boolean isAllowCoreThreadsTimeout() {
        return getParent() != null && allowCoreThreadsTimeout == null ? getParent().isAllowCoreThreadsTimeout() : allowCoreThreadsTimeout;
    }

    /**
     * @param allowCoreThreadsTimeout
     *            the allowCoreThreadsTimeout to set
     */
    public void setAllowCoreThreadsTimeout(boolean allowCoreThreadsTimeout) {
        this.allowCoreThreadsTimeout = allowCoreThreadsTimeout;
    }

    /**
     * @return if synchronous tasks should bypass the executor
     */
    public boolean isBypassIfSynchronous() {
        return getParent() != null && bypassIfSynchronous == null ? getParent().isBypassIfSynchronous() : bypassIfSynchronous;
    }

    /**
     * @param bypassIfSynchronous if synchronous tasks should bypass the executor
     */
    public void setBypassIfSynchronous(boolean bypassIfSynchronous) {
        this.bypassIfSynchronous = bypassIfSynchronous;
    }

    public ExecutorConfig getParent() {
        return parent;
    }

    public void setParent(ExecutorConfig parent) {
        this.parent = parent;
    }
}
