/*
 * Copyright 2011 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.drools.planner.core.move.generic;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.drools.planner.core.domain.variable.PlanningVariableDescriptor;
import org.drools.planner.core.move.Move;
import org.drools.planner.core.score.director.ScoreDirector;

/**
 * Non-cacheable
 */
public class GenericSwapPillarMove implements Move {

    private final Collection<PlanningVariableDescriptor> planningVariableDescriptors;

    private final List<Object> leftPlanningEntityList;

    private final List<Object> rightPlanningEntityList;

    public GenericSwapPillarMove(Collection<PlanningVariableDescriptor> planningVariableDescriptors,
            List<Object> leftPlanningEntityList, List<Object> rightPlanningEntityList) {
        this.planningVariableDescriptors = planningVariableDescriptors;
        this.leftPlanningEntityList = leftPlanningEntityList;
        this.rightPlanningEntityList = rightPlanningEntityList;
    }

    public boolean isMoveDoable(ScoreDirector scoreDirector) {
        for (PlanningVariableDescriptor planningVariableDescriptor : planningVariableDescriptors) {
            Object leftValue = planningVariableDescriptor.getValue(leftPlanningEntityList.get(0));
            Object rightValue = planningVariableDescriptor.getValue(rightPlanningEntityList.get(0));
            if (!ObjectUtils.equals(leftValue, rightValue)) {
                return true;
            }
        }
        return false;
    }

    public Move createUndoMove(ScoreDirector scoreDirector) {
        return new GenericSwapPillarMove(planningVariableDescriptors,
                rightPlanningEntityList, leftPlanningEntityList);
    }

    public void doMove(ScoreDirector scoreDirector) {
        for (PlanningVariableDescriptor planningVariableDescriptor : planningVariableDescriptors) {
            Object leftValue = planningVariableDescriptor.getValue(leftPlanningEntityList.get(0));
            Object rightValue = planningVariableDescriptor.getValue(rightPlanningEntityList.get(0));
            if (!ObjectUtils.equals(leftValue, rightValue)) {
                for (Object leftPlanningEntity : leftPlanningEntityList) {
                    scoreDirector.beforeVariableChanged(leftPlanningEntity, planningVariableDescriptor.getVariableName());
                    planningVariableDescriptor.setValue(leftPlanningEntity, rightValue);
                    scoreDirector.afterVariableChanged(leftPlanningEntity, planningVariableDescriptor.getVariableName());
                }
                for (Object rightPlanningEntity : rightPlanningEntityList) {
                    scoreDirector.beforeVariableChanged(rightPlanningEntity, planningVariableDescriptor.getVariableName());
                    planningVariableDescriptor.setValue(rightPlanningEntity, leftValue);
                    scoreDirector.afterVariableChanged(rightPlanningEntity, planningVariableDescriptor.getVariableName());
                }
            }
        }
    }

    public Collection<? extends Object> getPlanningEntities() {
        List<Object> entities = new ArrayList<Object>(
                leftPlanningEntityList.size() + rightPlanningEntityList.size());
        entities.addAll(leftPlanningEntityList);
        entities.addAll(rightPlanningEntityList);
        return entities;
    }

    public Collection<? extends Object> getPlanningValues() {
        List<Object> values = new ArrayList<Object>(planningVariableDescriptors.size() * 2);
        for (PlanningVariableDescriptor planningVariableDescriptor : planningVariableDescriptors) {
            values.add(planningVariableDescriptor.getValue(leftPlanningEntityList.get(0)));
            values.add(planningVariableDescriptor.getValue(rightPlanningEntityList.get(0)));
        }
        return values;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        } else if (o instanceof GenericSwapPillarMove) {
            GenericSwapPillarMove other = (GenericSwapPillarMove) o;
            return new EqualsBuilder()
                    .append(leftPlanningEntityList, other.leftPlanningEntityList)
                    .append(rightPlanningEntityList, other.rightPlanningEntityList)
                    .isEquals();
        } else {
            return false;
        }
    }

    public int hashCode() {
        return new HashCodeBuilder()
                .append(leftPlanningEntityList)
                .append(rightPlanningEntityList)
                .toHashCode();
    }

    public String toString() {
        return leftPlanningEntityList + " <=> " + rightPlanningEntityList;
    }

}
