/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.spdy.http;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;
import org.eclipse.jetty.spdy.api.Headers;
import org.eclipse.jetty.spdy.api.Stream;
import org.eclipse.jetty.spdy.http.HTTPSPDYHeader;
import org.eclipse.jetty.spdy.http.PushStrategy;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class ReferrerPushStrategy
implements PushStrategy {
    private static final Logger logger = Log.getLogger(ReferrerPushStrategy.class);
    private final ConcurrentMap<String, MainResource> mainResources = new ConcurrentHashMap<String, MainResource>();
    private final Set<Pattern> pushRegexps = new HashSet<Pattern>();
    private final Set<String> pushContentTypes = new HashSet<String>();
    private final Set<Pattern> allowedPushOrigins = new HashSet<Pattern>();
    private volatile int maxAssociatedResources = 32;
    private volatile int referrerPushPeriod = 5000;

    public ReferrerPushStrategy() {
        this(Arrays.asList(".*\\.css", ".*\\.js", ".*\\.png", ".*\\.jpeg", ".*\\.jpg", ".*\\.gif", ".*\\.ico"));
    }

    public ReferrerPushStrategy(List<String> pushRegexps) {
        this(pushRegexps, Arrays.asList("text/css", "text/javascript", "application/javascript", "application/x-javascript", "image/png", "image/x-png", "image/jpeg", "image/gif", "image/x-icon", "image/vnd.microsoft.icon"));
    }

    public ReferrerPushStrategy(List<String> pushRegexps, List<String> pushContentTypes) {
        this(pushRegexps, pushContentTypes, Collections.emptyList());
    }

    public ReferrerPushStrategy(List<String> pushRegexps, List<String> pushContentTypes, List<String> allowedPushOrigins) {
        for (String pushRegexp : pushRegexps) {
            this.pushRegexps.add(Pattern.compile(pushRegexp));
        }
        this.pushContentTypes.addAll(pushContentTypes);
        for (String allowedPushOrigin : allowedPushOrigins) {
            this.allowedPushOrigins.add(Pattern.compile(allowedPushOrigin.replace(".", "\\.").replace("*", ".*")));
        }
    }

    public int getMaxAssociatedResources() {
        return this.maxAssociatedResources;
    }

    public void setMaxAssociatedResources(int maxAssociatedResources) {
        this.maxAssociatedResources = maxAssociatedResources;
    }

    public int getReferrerPushPeriod() {
        return this.referrerPushPeriod;
    }

    public void setReferrerPushPeriod(int referrerPushPeriod) {
        this.referrerPushPeriod = referrerPushPeriod;
    }

    @Override
    public Set<String> apply(Stream stream, Headers requestHeaders, Headers responseHeaders) {
        Set<String> result = Collections.emptySet();
        short version = stream.getSession().getVersion();
        if (!this.isIfModifiedSinceHeaderPresent(requestHeaders) && this.isValidMethod(requestHeaders.get(HTTPSPDYHeader.METHOD.name(version)).value())) {
            Headers.Header referrerHeader;
            String scheme = requestHeaders.get(HTTPSPDYHeader.SCHEME.name(version)).value();
            String host = requestHeaders.get(HTTPSPDYHeader.HOST.name(version)).value();
            String origin = scheme + "://" + host;
            String url = requestHeaders.get(HTTPSPDYHeader.URI.name(version)).value();
            String absoluteURL = origin + url;
            logger.debug("Applying push strategy for {}", new Object[]{absoluteURL});
            if (this.isMainResource(url, responseHeaders)) {
                MainResource mainResource = this.getOrCreateMainResource(absoluteURL);
                result = mainResource.getResources();
            } else if (this.isPushResource(url, responseHeaders) && (referrerHeader = requestHeaders.get("referer")) != null) {
                Set<String> pushResources;
                String referrer = referrerHeader.value();
                MainResource mainResource = (MainResource)this.mainResources.get(referrer);
                if (mainResource == null) {
                    mainResource = this.getOrCreateMainResource(referrer);
                }
                if (!(pushResources = mainResource.getResources()).contains(url)) {
                    mainResource.addResource(url, origin, referrer);
                } else {
                    result = this.getPushResources(absoluteURL);
                }
            }
            logger.debug("Pushing {} resources for {}: {}", new Object[]{result.size(), absoluteURL, result});
        }
        return result;
    }

    private Set<String> getPushResources(String absoluteURL) {
        Set<String> result = Collections.emptySet();
        if (this.mainResources.get(absoluteURL) != null) {
            result = ((MainResource)this.mainResources.get(absoluteURL)).getResources();
        }
        return result;
    }

    private MainResource getOrCreateMainResource(String absoluteURL) {
        MainResource mainResource = (MainResource)this.mainResources.get(absoluteURL);
        if (mainResource == null) {
            logger.debug("Creating new main resource for {}", new Object[]{absoluteURL});
            MainResource value = new MainResource(absoluteURL);
            mainResource = this.mainResources.putIfAbsent(absoluteURL, value);
            if (mainResource == null) {
                mainResource = value;
            }
        }
        return mainResource;
    }

    private boolean isIfModifiedSinceHeaderPresent(Headers headers) {
        return headers.get("if-modified-since") != null;
    }

    private boolean isValidMethod(String method) {
        return "GET".equalsIgnoreCase(method);
    }

    private boolean isMainResource(String url, Headers responseHeaders) {
        return !this.isPushResource(url, responseHeaders);
    }

    private boolean isPushResource(String url, Headers responseHeaders) {
        for (Pattern pushRegexp : this.pushRegexps) {
            if (!pushRegexp.matcher(url).matches()) continue;
            Headers.Header header = responseHeaders.get("content-type");
            if (header == null) {
                return true;
            }
            String contentType = header.value().toLowerCase();
            for (String pushContentType : this.pushContentTypes) {
                if (!contentType.startsWith(pushContentType)) continue;
                return true;
            }
        }
        return false;
    }

    private class MainResource {
        private final String name;
        private final Set<String> resources = Collections.newSetFromMap(new ConcurrentHashMap());
        private final AtomicLong firstResourceAdded = new AtomicLong(-1L);

        private MainResource(String name) {
            this.name = name;
        }

        public boolean addResource(String url, String origin, String referrer) {
            this.firstResourceAdded.compareAndSet(-1L, System.nanoTime());
            long delay = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - this.firstResourceAdded.get());
            if (!referrer.startsWith(origin) && !this.isPushOriginAllowed(origin)) {
                logger.debug("Skipped store of push metadata {} for {}: Origin: {} doesn't match or origin not allowed", new Object[]{url, this.name, origin});
                return false;
            }
            if (this.resources.size() >= ReferrerPushStrategy.this.maxAssociatedResources) {
                logger.debug("Skipped store of push metadata {} for {}: max associated resources ({}) reached", new Object[]{url, this.name, ReferrerPushStrategy.this.maxAssociatedResources});
                return false;
            }
            if (delay > (long)ReferrerPushStrategy.this.referrerPushPeriod) {
                logger.debug("Delay: {}ms longer than referrerPushPeriod: {}ms. Not adding resource: {} for: {}", new Object[]{delay, ReferrerPushStrategy.this.referrerPushPeriod, url, this.name});
                return false;
            }
            logger.debug("Adding resource: {} for: {} with delay: {}ms.", new Object[]{url, this.name, delay});
            this.resources.add(url);
            return true;
        }

        public Set<String> getResources() {
            return Collections.unmodifiableSet(this.resources);
        }

        public String toString() {
            return "MainResource: " + this.name + " associated resources:" + this.resources.size();
        }

        private boolean isPushOriginAllowed(String origin) {
            for (Pattern allowedPushOrigin : ReferrerPushStrategy.this.allowedPushOrigins) {
                if (!allowedPushOrigin.matcher(origin).matches()) continue;
                return true;
            }
            return false;
        }
    }
}

