/*
 * Decompiled with CFR 0.152.
 */
package org.postgresql.jdbc;

import java.sql.Date;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.chrono.IsoEra;
import java.time.temporal.ChronoField;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.SimpleTimeZone;
import java.util.TimeZone;
import org.postgresql.util.ByteConverter;
import org.postgresql.util.GT;
import org.postgresql.util.PSQLException;
import org.postgresql.util.PSQLState;

public class TimestampUtils {
    private static final int ONEDAY = 86400000;
    private static final char[] ZEROS = new char[]{'0', '0', '0', '0', '0', '0', '0', '0', '0'};
    private static final char[][] NUMBERS = new char[64][];
    private final StringBuilder sbuf = new StringBuilder();
    private final Calendar calendarWithUserTz = new GregorianCalendar();
    private final TimeZone utcTz = TimeZone.getTimeZone("UTC");
    private Calendar calCache;
    private int calCacheZone;
    private final boolean min74;
    private final boolean min82;
    private final boolean usesDouble;

    TimestampUtils(boolean min74, boolean min82, boolean usesDouble) {
        this.min74 = min74;
        this.min82 = min82;
        this.usesDouble = usesDouble;
    }

    private Calendar getCalendar(int sign, int hr, int min, int sec) {
        int rawOffset = sign * (((hr * 60 + min) * 60 + sec) * 1000);
        if (this.calCache != null && this.calCacheZone == rawOffset) {
            return this.calCache;
        }
        StringBuilder zoneID = new StringBuilder("GMT");
        zoneID.append(sign < 0 ? (char)'-' : '+');
        if (hr < 10) {
            zoneID.append('0');
        }
        zoneID.append(hr);
        if (min < 10) {
            zoneID.append('0');
        }
        zoneID.append(min);
        if (sec < 10) {
            zoneID.append('0');
        }
        zoneID.append(sec);
        SimpleTimeZone syntheticTZ = new SimpleTimeZone(rawOffset, zoneID.toString());
        this.calCache = new GregorianCalendar(syntheticTZ);
        this.calCacheZone = rawOffset;
        return this.calCache;
    }

    private ParsedTimestamp parseBackendTimestamp(String str) throws SQLException {
        char[] s = str.toCharArray();
        int slen = s.length;
        ParsedTimestamp result = new ParsedTimestamp();
        try {
            char sep;
            int start = TimestampUtils.skipWhitespace(s, 0);
            int end = TimestampUtils.firstNonDigit(s, start);
            if (TimestampUtils.charAt(s, end) == '-') {
                result.hasDate = true;
                result.year = TimestampUtils.number(s, start, end);
                start = end + 1;
                end = TimestampUtils.firstNonDigit(s, start);
                result.month = TimestampUtils.number(s, start, end);
                sep = TimestampUtils.charAt(s, end);
                if (sep != '-') {
                    throw new NumberFormatException("Expected date to be dash-separated, got '" + sep + "'");
                }
                start = end + 1;
                end = TimestampUtils.firstNonDigit(s, start);
                result.day = TimestampUtils.number(s, start, end);
                start = TimestampUtils.skipWhitespace(s, end);
            }
            if (Character.isDigit(TimestampUtils.charAt(s, start))) {
                result.hasTime = true;
                end = TimestampUtils.firstNonDigit(s, start);
                result.hour = TimestampUtils.number(s, start, end);
                sep = TimestampUtils.charAt(s, end);
                if (sep != ':') {
                    throw new NumberFormatException("Expected time to be colon-separated, got '" + sep + "'");
                }
                start = end + 1;
                end = TimestampUtils.firstNonDigit(s, start);
                result.minute = TimestampUtils.number(s, start, end);
                sep = TimestampUtils.charAt(s, end);
                if (sep != ':') {
                    throw new NumberFormatException("Expected time to be colon-separated, got '" + sep + "'");
                }
                start = end + 1;
                end = TimestampUtils.firstNonDigit(s, start);
                result.second = TimestampUtils.number(s, start, end);
                start = end;
                if (TimestampUtils.charAt(s, start) == '.') {
                    end = TimestampUtils.firstNonDigit(s, start + 1);
                    int num = TimestampUtils.number(s, start + 1, end);
                    for (int numlength = end - (start + 1); numlength < 9; ++numlength) {
                        num *= 10;
                    }
                    result.nanos = num;
                    start = end;
                }
                start = TimestampUtils.skipWhitespace(s, start);
            }
            if ((sep = TimestampUtils.charAt(s, start)) == '-' || sep == '+') {
                int tzmin;
                int tzsign = sep == '-' ? -1 : 1;
                end = TimestampUtils.firstNonDigit(s, start + 1);
                int tzhr = TimestampUtils.number(s, start + 1, end);
                start = end;
                sep = TimestampUtils.charAt(s, start);
                if (sep == ':') {
                    end = TimestampUtils.firstNonDigit(s, start + 1);
                    tzmin = TimestampUtils.number(s, start + 1, end);
                    start = end;
                } else {
                    tzmin = 0;
                }
                int tzsec = 0;
                if (this.min82 && (sep = TimestampUtils.charAt(s, start)) == ':') {
                    end = TimestampUtils.firstNonDigit(s, start + 1);
                    tzsec = TimestampUtils.number(s, start + 1, end);
                    start = end;
                }
                result.tz = this.getCalendar(tzsign, tzhr, tzmin, tzsec);
                start = TimestampUtils.skipWhitespace(s, start);
            }
            if (result.hasDate && start < slen) {
                String eraString = new String(s, start, slen - start);
                if (eraString.startsWith("AD")) {
                    result.era = 1;
                    start += 2;
                } else if (eraString.startsWith("BC")) {
                    result.era = 0;
                    start += 2;
                }
            }
            if (start < slen) {
                throw new NumberFormatException("Trailing junk on timestamp: '" + new String(s, start, slen - start) + "'");
            }
            if (!result.hasTime && !result.hasDate) {
                throw new NumberFormatException("Timestamp has neither date nor time");
            }
        }
        catch (NumberFormatException nfe) {
            throw new PSQLException(GT.tr("Bad value for type timestamp/date/time: {1}", new Object[]{str}), PSQLState.BAD_DATETIME_FORMAT, (Throwable)nfe);
        }
        return result;
    }

    public synchronized Timestamp toTimestamp(Calendar cal, String s) throws SQLException {
        if (s == null) {
            return null;
        }
        int slen = s.length();
        if (slen == 8 && s.equals("infinity")) {
            return new Timestamp(9223372036825200000L);
        }
        if (slen == 9 && s.equals("-infinity")) {
            return new Timestamp(-9223372036832400000L);
        }
        ParsedTimestamp ts = this.parseBackendTimestamp(s);
        Calendar useCal = ts.tz != null ? ts.tz : this.setupCalendar(cal);
        useCal.set(0, ts.era);
        useCal.set(1, ts.year);
        useCal.set(2, ts.month - 1);
        useCal.set(5, ts.day);
        useCal.set(11, ts.hour);
        useCal.set(12, ts.minute);
        useCal.set(13, ts.second);
        useCal.set(14, 0);
        Timestamp result = new Timestamp(useCal.getTimeInMillis());
        result.setNanos(ts.nanos);
        return result;
    }

    public LocalDateTime toLocalDateTime(String s) throws SQLException {
        if (s == null) {
            return null;
        }
        int slen = s.length();
        if (slen == 8 && s.equals("infinity")) {
            return LocalDateTime.MAX;
        }
        if (slen == 9 && s.equals("-infinity")) {
            return LocalDateTime.MIN;
        }
        ParsedTimestamp ts = this.parseBackendTimestamp(s);
        LocalDateTime result = LocalDateTime.of(ts.year, ts.month, ts.day, ts.hour, ts.minute, ts.second, ts.nanos);
        if (ts.era == 0) {
            return result.with(ChronoField.ERA, IsoEra.BCE.getValue());
        }
        return result;
    }

    public synchronized Time toTime(Calendar cal, String s) throws SQLException {
        Timestamp timestamp = this.toTimestamp(cal, s);
        if (timestamp == null) {
            return null;
        }
        long millis = timestamp.getTime();
        if (millis <= -9223372036832400000L || millis >= 9223372036825200000L) {
            throw new PSQLException(GT.tr("Infinite value found for timestamp/date. This cannot be represented as time."), PSQLState.DATETIME_OVERFLOW);
        }
        return this.convertToTime(timestamp.getTime(), cal == null ? null : cal.getTimeZone());
    }

    public synchronized Date toDate(Calendar cal, String s) throws SQLException {
        Timestamp timestamp = this.toTimestamp(cal, s);
        if (timestamp == null) {
            return null;
        }
        return this.convertToDate(timestamp.getTime(), cal == null ? null : cal.getTimeZone());
    }

    private Calendar setupCalendar(Calendar cal) {
        Calendar tmp = this.calendarWithUserTz;
        tmp.setTimeZone(cal == null ? TimestampUtils.getDefaultTz() : cal.getTimeZone());
        return tmp;
    }

    public synchronized String toString(Calendar cal, Timestamp x) {
        if (x.getTime() == 9223372036825200000L) {
            return "infinity";
        }
        if (x.getTime() == -9223372036832400000L) {
            return "-infinity";
        }
        cal = this.setupCalendar(cal);
        cal.setTime(x);
        this.sbuf.setLength(0);
        TimestampUtils.appendDate(this.sbuf, cal);
        this.sbuf.append(' ');
        TimestampUtils.appendTime(this.sbuf, cal, x.getNanos());
        this.appendTimeZone(this.sbuf, cal);
        TimestampUtils.appendEra(this.sbuf, cal);
        return this.sbuf.toString();
    }

    public synchronized String toString(Calendar cal, Date x) {
        if (x.getTime() == 9223372036825200000L) {
            return "infinity";
        }
        if (x.getTime() == -9223372036832400000L) {
            return "-infinity";
        }
        cal = this.setupCalendar(cal);
        cal.setTime(x);
        this.sbuf.setLength(0);
        TimestampUtils.appendDate(this.sbuf, cal);
        TimestampUtils.appendEra(this.sbuf, cal);
        this.appendTimeZone(this.sbuf, cal);
        return this.sbuf.toString();
    }

    public synchronized String toString(Calendar cal, Time x) {
        cal = this.setupCalendar(cal);
        cal.setTime(x);
        this.sbuf.setLength(0);
        TimestampUtils.appendTime(this.sbuf, cal, cal.get(14) * 1000000);
        if (this.min74) {
            this.appendTimeZone(this.sbuf, cal);
        }
        return this.sbuf.toString();
    }

    private static void appendDate(StringBuilder sb, Calendar cal) {
        int l_year = cal.get(1);
        int l_month = cal.get(2) + 1;
        int l_day = cal.get(5);
        TimestampUtils.appendDate(sb, l_year, l_month, l_day);
    }

    private static void appendDate(StringBuilder sb, int year, int month, int day) {
        int prevLength = sb.length();
        sb.append(year);
        int leadingZerosForYear = 4 - (sb.length() - prevLength);
        if (leadingZerosForYear > 0) {
            sb.insert(prevLength, ZEROS, 0, leadingZerosForYear);
        }
        sb.append('-');
        sb.append(NUMBERS[month]);
        sb.append('-');
        sb.append(NUMBERS[day]);
    }

    private static void appendTime(StringBuilder sb, Calendar cal, int nanos) {
        int hours = cal.get(11);
        int minutes = cal.get(12);
        int seconds = cal.get(13);
        TimestampUtils.appendTime(sb, hours, minutes, seconds, nanos);
    }

    private static void appendTime(StringBuilder sb, int hours, int minutes, int seconds, int nanos) {
        sb.append(NUMBERS[hours]);
        sb.append(':');
        sb.append(NUMBERS[minutes]);
        sb.append(':');
        sb.append(NUMBERS[seconds]);
        sb.append('.');
        int len = sb.length();
        sb.append(nanos / 1000);
        int needZeros = 6 - (sb.length() - len);
        if (needZeros > 0) {
            sb.insert(len, ZEROS, 0, needZeros);
        }
    }

    private void appendTimeZone(StringBuilder sb, Calendar cal) {
        int offset = (cal.get(15) + cal.get(16)) / 1000;
        this.appendTimeZone(sb, offset);
    }

    private void appendTimeZone(StringBuilder sb, int offset) {
        int absoff = Math.abs(offset);
        int hours = absoff / 60 / 60;
        int mins = (absoff - hours * 60 * 60) / 60;
        int secs = absoff - hours * 60 * 60 - mins * 60;
        sb.append(offset >= 0 ? " +" : " -");
        sb.append(NUMBERS[hours]);
        sb.append(':');
        sb.append(NUMBERS[mins]);
        if (this.min82) {
            sb.append(':');
            sb.append(NUMBERS[secs]);
        }
    }

    private static void appendEra(StringBuilder sb, Calendar cal) {
        if (cal.get(0) == 0) {
            sb.append(" BC");
        }
    }

    public synchronized String toString(LocalDate localDate) {
        if (LocalDate.MAX.equals(localDate)) {
            return "infinity";
        }
        if (LocalDate.MIN.equals(localDate)) {
            return "-infinity";
        }
        this.sbuf.setLength(0);
        TimestampUtils.appendDate(this.sbuf, localDate);
        TimestampUtils.appendEra(this.sbuf, localDate);
        return this.sbuf.toString();
    }

    public synchronized String toString(LocalTime localTime) {
        if (LocalTime.MAX.equals(localTime)) {
            return "infinity";
        }
        if (LocalTime.MIN.equals(localTime)) {
            return "-infinity";
        }
        this.sbuf.setLength(0);
        TimestampUtils.appendTime(this.sbuf, localTime);
        return this.sbuf.toString();
    }

    public synchronized String toString(OffsetDateTime offsetDateTime) {
        if (OffsetDateTime.MAX.equals(offsetDateTime)) {
            return "infinity";
        }
        if (OffsetDateTime.MIN.equals(offsetDateTime)) {
            return "-infinity";
        }
        this.sbuf.setLength(0);
        LocalDateTime localDateTime = offsetDateTime.toLocalDateTime();
        LocalDate localDate = localDateTime.toLocalDate();
        TimestampUtils.appendDate(this.sbuf, localDate);
        this.sbuf.append(' ');
        TimestampUtils.appendTime(this.sbuf, localDateTime.toLocalTime());
        this.appendTimeZone(this.sbuf, offsetDateTime.getOffset());
        TimestampUtils.appendEra(this.sbuf, localDate);
        return this.sbuf.toString();
    }

    public synchronized String toString(LocalDateTime localDateTime) {
        if (LocalDateTime.MAX.equals(localDateTime)) {
            return "infinity";
        }
        if (LocalDateTime.MIN.equals(localDateTime)) {
            return "-infinity";
        }
        this.sbuf.setLength(0);
        LocalDate localDate = localDateTime.toLocalDate();
        TimestampUtils.appendDate(this.sbuf, localDate);
        this.sbuf.append(' ');
        TimestampUtils.appendTime(this.sbuf, localDateTime.toLocalTime());
        TimestampUtils.appendEra(this.sbuf, localDate);
        return this.sbuf.toString();
    }

    private static void appendDate(StringBuilder sb, LocalDate localDate) {
        int year = Math.abs(localDate.getYear());
        int month = localDate.getMonthValue();
        int day = localDate.getDayOfMonth();
        TimestampUtils.appendDate(sb, year, month, day);
    }

    private static void appendTime(StringBuilder sb, LocalTime localTime) {
        int hours = localTime.getHour();
        int minutes = localTime.getMinute();
        int seconds = localTime.getSecond();
        int nanos = localTime.getNano();
        TimestampUtils.appendTime(sb, hours, minutes, seconds, nanos);
    }

    private void appendTimeZone(StringBuilder sb, ZoneOffset offset) {
        int offsetSeconds = offset.getTotalSeconds();
        this.appendTimeZone(sb, offsetSeconds);
    }

    private static void appendEra(StringBuilder sb, LocalDate localDate) {
        if (localDate.get(ChronoField.ERA) == IsoEra.BCE.getValue()) {
            sb.append(" BC");
        }
    }

    private static int skipWhitespace(char[] s, int start) {
        int slen = s.length;
        for (int i = start; i < slen; ++i) {
            if (Character.isSpace(s[i])) continue;
            return i;
        }
        return slen;
    }

    private static int firstNonDigit(char[] s, int start) {
        int slen = s.length;
        for (int i = start; i < slen; ++i) {
            if (Character.isDigit(s[i])) continue;
            return i;
        }
        return slen;
    }

    private static int number(char[] s, int start, int end) {
        if (start >= end) {
            throw new NumberFormatException();
        }
        int n = 0;
        for (int i = start; i < end; ++i) {
            n = 10 * n + (s[i] - 48);
        }
        return n;
    }

    private static char charAt(char[] s, int pos) {
        if (pos >= 0 && pos < s.length) {
            return s[pos];
        }
        return '\u0000';
    }

    public Date toDateBin(TimeZone tz, byte[] bytes) throws PSQLException {
        if (bytes.length != 4) {
            throw new PSQLException(GT.tr("Unsupported binary encoding of {0}.", "date"), PSQLState.BAD_DATETIME_FORMAT);
        }
        int days = ByteConverter.int4(bytes, 0);
        if (tz == null) {
            tz = TimestampUtils.getDefaultTz();
        }
        long secs = TimestampUtils.toJavaSecs((long)days * 86400L);
        long millis = secs * 1000L;
        if ((millis = this.guessTimestamp(millis, tz)) <= -185543533774800000L) {
            millis = -9223372036832400000L;
        } else if (millis >= 185543533774800000L) {
            millis = 9223372036825200000L;
        }
        return new Date(millis);
    }

    private static TimeZone getDefaultTz() {
        return TimeZone.getDefault();
    }

    public Time toTimeBin(TimeZone tz, byte[] bytes) throws PSQLException {
        long millis;
        if (bytes.length != 8 && bytes.length != 12) {
            throw new PSQLException(GT.tr("Unsupported binary encoding of {0}.", "time"), PSQLState.BAD_DATETIME_FORMAT);
        }
        if (this.usesDouble) {
            double time = ByteConverter.float8(bytes, 0);
            millis = (long)(time * 1000.0);
        } else {
            long time = ByteConverter.int8(bytes, 0);
            millis = time / 1000L;
        }
        if (bytes.length == 12) {
            int timeOffset = ByteConverter.int4(bytes, 8);
            millis -= (long)(timeOffset *= -1000);
        } else {
            if (tz == null) {
                tz = TimestampUtils.getDefaultTz();
            }
            millis = this.guessTimestamp(millis, tz);
        }
        return this.convertToTime(millis, tz);
    }

    public Timestamp toTimestampBin(TimeZone tz, byte[] bytes, boolean timestamptz) throws PSQLException {
        ParsedBinaryTimestamp parsedTimestamp = this.toParsedTimestampBin(tz, bytes, timestamptz);
        if (parsedTimestamp.infinity == Infinity.POSITIVE) {
            return new Timestamp(9223372036825200000L);
        }
        if (parsedTimestamp.infinity == Infinity.NEGATIVE) {
            return new Timestamp(-9223372036832400000L);
        }
        Timestamp ts = new Timestamp(parsedTimestamp.millis);
        ts.setNanos(parsedTimestamp.nanos);
        return ts;
    }

    private ParsedBinaryTimestamp toParsedTimestampBin(TimeZone tz, byte[] bytes, boolean timestamptz) throws PSQLException {
        int nanos;
        long secs;
        if (bytes.length != 8) {
            throw new PSQLException(GT.tr("Unsupported binary encoding of {0}.", "timestamp"), PSQLState.BAD_DATETIME_FORMAT);
        }
        if (this.usesDouble) {
            double time = ByteConverter.float8(bytes, 0);
            if (time == Double.POSITIVE_INFINITY) {
                ParsedBinaryTimestamp ts = new ParsedBinaryTimestamp();
                ts.infinity = Infinity.POSITIVE;
                return ts;
            }
            if (time == Double.NEGATIVE_INFINITY) {
                ParsedBinaryTimestamp ts = new ParsedBinaryTimestamp();
                ts.infinity = Infinity.NEGATIVE;
                return ts;
            }
            secs = (long)time;
            nanos = (int)((time - (double)secs) * 1000000.0);
        } else {
            long time = ByteConverter.int8(bytes, 0);
            if (time == Long.MAX_VALUE) {
                ParsedBinaryTimestamp ts = new ParsedBinaryTimestamp();
                ts.infinity = Infinity.POSITIVE;
                return ts;
            }
            if (time == Long.MIN_VALUE) {
                ParsedBinaryTimestamp ts = new ParsedBinaryTimestamp();
                ts.infinity = Infinity.NEGATIVE;
                return ts;
            }
            secs = time / 1000000L;
            nanos = (int)(time - secs * 1000000L);
        }
        if (nanos < 0) {
            --secs;
            nanos += 1000000;
        }
        nanos *= 1000;
        secs = TimestampUtils.toJavaSecs(secs);
        long millis = secs * 1000L;
        if (!timestamptz) {
            millis = this.guessTimestamp(millis, tz);
        }
        ParsedBinaryTimestamp ts = new ParsedBinaryTimestamp();
        ts.millis = millis;
        ts.nanos = nanos;
        return ts;
    }

    public LocalDateTime toLocalDateTimeBin(byte[] bytes) throws PSQLException {
        ParsedBinaryTimestamp parsedTimestamp = this.toParsedTimestampBin(null, bytes, true);
        if (parsedTimestamp.infinity == Infinity.POSITIVE) {
            return LocalDateTime.MAX;
        }
        if (parsedTimestamp.infinity == Infinity.NEGATIVE) {
            return LocalDateTime.MAX;
        }
        return LocalDateTime.ofEpochSecond(parsedTimestamp.millis / 1000L, parsedTimestamp.nanos, ZoneOffset.UTC);
    }

    private long guessTimestamp(long millis, TimeZone tz) {
        if (tz == null) {
            tz = TimestampUtils.getDefaultTz();
        }
        if (TimestampUtils.isSimpleTimeZone(tz.getID())) {
            return millis - (long)tz.getRawOffset();
        }
        Calendar cal = this.calendarWithUserTz;
        cal.setTimeZone(this.utcTz);
        cal.setTimeInMillis(millis);
        int era = cal.get(0);
        int year = cal.get(1);
        int month = cal.get(2);
        int day = cal.get(5);
        int hour = cal.get(11);
        int min = cal.get(12);
        int sec = cal.get(13);
        int ms = cal.get(14);
        cal.setTimeZone(tz);
        cal.set(0, era);
        cal.set(1, year);
        cal.set(2, month);
        cal.set(5, day);
        cal.set(11, hour);
        cal.set(12, min);
        cal.set(13, sec);
        cal.set(14, ms);
        return cal.getTimeInMillis();
    }

    private static boolean isSimpleTimeZone(String id) {
        return id.startsWith("GMT") || id.startsWith("UTC");
    }

    public Date convertToDate(long millis, TimeZone tz) {
        if (millis <= -9223372036832400000L || millis >= 9223372036825200000L) {
            return new Date(millis);
        }
        if (tz == null) {
            tz = TimestampUtils.getDefaultTz();
        }
        if (TimestampUtils.isSimpleTimeZone(tz.getID())) {
            int offset = tz.getRawOffset();
            millis += (long)offset;
            millis = millis / 86400000L * 86400000L;
            return new Date(millis -= (long)offset);
        }
        Calendar cal = this.calendarWithUserTz;
        cal.setTimeZone(tz);
        cal.setTimeInMillis(millis);
        cal.set(11, 0);
        cal.set(12, 0);
        cal.set(13, 0);
        cal.set(14, 0);
        return new Date(cal.getTimeInMillis());
    }

    public Time convertToTime(long millis, TimeZone tz) {
        if (tz == null) {
            tz = TimestampUtils.getDefaultTz();
        }
        if (TimestampUtils.isSimpleTimeZone(tz.getID())) {
            int offset = tz.getRawOffset();
            millis += (long)offset;
            millis %= 86400000L;
            return new Time(millis -= (long)offset);
        }
        Calendar cal = this.calendarWithUserTz;
        cal.setTimeZone(tz);
        cal.setTimeInMillis(millis);
        cal.set(0, 1);
        cal.set(1, 1970);
        cal.set(2, 0);
        cal.set(5, 1);
        return new Time(cal.getTimeInMillis());
    }

    public String timeToString(java.util.Date time) {
        long millis = time.getTime();
        if (millis <= -9223372036832400000L) {
            return "-infinity";
        }
        if (millis >= 9223372036825200000L) {
            return "infinity";
        }
        return time.toString();
    }

    private static long toJavaSecs(long secs) {
        if ((secs += 946684800L) < -12219292800L && (secs += 864000L) < -14825808000L) {
            int extraLeaps = (int)((secs + 14825808000L) / 3155760000L);
            --extraLeaps;
            extraLeaps -= extraLeaps / 4;
            secs += (long)extraLeaps * 86400L;
        }
        return secs;
    }

    private static long toPgSecs(long secs) {
        if ((secs -= 946684800L) < -13165977600L && (secs -= 864000L) < -15773356800L) {
            int years = (int)((secs + 15773356800L) / -3155823050L);
            ++years;
            years -= years / 4;
            secs += (long)(years * 86400);
        }
        return secs;
    }

    public void toBinDate(TimeZone tz, byte[] bytes, Date value) throws PSQLException {
        long millis = value.getTime();
        if (tz == null) {
            tz = TimestampUtils.getDefaultTz();
        }
        millis += (long)tz.getOffset(millis);
        long secs = TimestampUtils.toPgSecs(millis / 1000L);
        ByteConverter.int4(bytes, 0, (int)(secs / 86400L));
    }

    static {
        for (int i = 0; i < NUMBERS.length; ++i) {
            TimestampUtils.NUMBERS[i] = Integer.toString(i).toCharArray();
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    static enum Infinity {
        POSITIVE,
        NEGATIVE;

    }

    private static class ParsedBinaryTimestamp {
        Infinity infinity = null;
        long millis = 0L;
        int nanos = 0;

        private ParsedBinaryTimestamp() {
        }
    }

    private static class ParsedTimestamp {
        boolean hasDate = false;
        int era = 1;
        int year = 1970;
        int month = 1;
        boolean hasTime = false;
        int day = 1;
        int hour = 0;
        int minute = 0;
        int second = 0;
        int nanos = 0;
        Calendar tz = null;

        private ParsedTimestamp() {
        }
    }
}

