/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.cache.infinispan.query;

import javax.transaction.Transaction;
import javax.transaction.TransactionManager;
import javax.transaction.Status;
import javax.transaction.Synchronization;

import org.hibernate.HibernateException;
import org.hibernate.cache.CacheException;
import org.hibernate.cache.infinispan.impl.BaseTransactionalDataRegion;
import org.hibernate.cache.infinispan.util.Caches;
import org.hibernate.cache.spi.QueryResultsRegion;
import org.hibernate.cache.spi.RegionFactory;
import org.hibernate.engine.spi.SessionImplementor;
import org.hibernate.jdbc.WorkExecutor;
import org.hibernate.jdbc.WorkExecutorVisitable;
import org.hibernate.resource.transaction.TransactionCoordinator;
import org.infinispan.AdvancedCache;
import org.infinispan.configuration.cache.TransactionConfiguration;
import org.infinispan.context.Flag;
import org.infinispan.transaction.TransactionMode;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Region for caching query results.
 *
 * @author Chris Bredesen
 * @author Galder Zamarreño
 * @since 3.5
 */
public class QueryResultsRegionImpl extends BaseTransactionalDataRegion implements QueryResultsRegion {
	private static final Log log = LogFactory.getLog( QueryResultsRegionImpl.class );

	private final AdvancedCache evictCache;
	private final AdvancedCache putCache;
	private final AdvancedCache getCache;
	private final ConcurrentMap<SessionImplementor, Map> transactionContext = new ConcurrentHashMap<SessionImplementor, Map>();
	private final boolean putCacheRequiresTransaction;

   /**
    * Query region constructor
    *
    * @param cache instance to store queries
    * @param name of the query region
    * @param factory for the query region
    */
	public QueryResultsRegionImpl(AdvancedCache cache, String name, RegionFactory factory) {
		super( cache, name, null, factory, null );
		// If Infinispan is using INVALIDATION for query cache, we don't want to propagate changes.
		// We use the Timestamps cache to manage invalidation
		final boolean localOnly = Caches.isInvalidationCache( cache );

		this.evictCache = localOnly ? Caches.localCache( cache ) : cache;

		this.putCache = localOnly ?
				Caches.failSilentWriteCache( cache, Flag.CACHE_MODE_LOCAL ) :
				Caches.failSilentWriteCache( cache );

		this.getCache = Caches.failSilentReadCache( cache );

		TransactionConfiguration transactionConfiguration = putCache.getCacheConfiguration().transaction();
		boolean transactional = transactionConfiguration.transactionMode() != TransactionMode.NON_TRANSACTIONAL;
		this.putCacheRequiresTransaction = transactional && !transactionConfiguration.autoCommit();
		// Since we execute the query update explicitly form transaction synchronization, the putCache does not need
		// to be transactional anymore (it had to be in the past to prevent revealing uncommitted changes).
		if (transactional) {
			log.warn("Use non-transactional query caches for best performance!");
		}
	}

	@Override
	public void evict(Object key) throws CacheException {
		for (Map map : transactionContext.values()) {
			map.remove(key);
		}
		evictCache.remove( key );
	}

	@Override
	public void evictAll() throws CacheException {
		transactionContext.clear();
		final Transaction tx = suspend();
		try {
			// Invalidate the local region and then go remote
			invalidateRegion();
			Caches.broadcastEvictAll( cache );
		}
		finally {
			resume( tx );
		}
	}

	@Override
	public Object get(SessionImplementor session, Object key) throws CacheException {
		// If the region is not valid, skip cache store to avoid going remote to retrieve the query.
		// The aim of this is to maintain same logic/semantics as when state transfer was configured.
		// TODO: Once https://issues.jboss.org/browse/ISPN-835 has been resolved, revert to state transfer and remove workaround
		boolean skipCacheStore = false;
		if ( !isValid() ) {
			skipCacheStore = true;
		}

		if ( !checkValid() ) {
			return null;
		}

		// In Infinispan get doesn't acquire any locks, so no need to suspend the tx.
		// In the past, when get operations acquired locks, suspending the tx was a way
		// to avoid holding locks that would prevent updates.
		// Add a zero (or low) timeout option so we don't block
		// waiting for tx's that did a put to commit
		Object result = null;
		Map map = transactionContext.get(session);
		if (map != null) {
			result = map.get(key);
		}
		if (result == null) {
			if ( skipCacheStore ) {
				result = getCache.withFlags( Flag.SKIP_CACHE_STORE ).get( key );
			}
			else {
				result = getCache.get( key );
			}
		}
		return result;
	}

	@Override
	@SuppressWarnings("unchecked")
	public void put(SessionImplementor session, Object key, Object value) throws CacheException {
		if ( checkValid() ) {
			// See HHH-7898: Even with FAIL_SILENTLY flag, failure to write in transaction
			// fails the whole transaction. It is an Infinispan quirk that cannot be fixed
			// ISPN-5356 tracks that. This is because if the transaction continued the
			// value could be committed on backup owners, including the failed operation,
			// and the result would not be consistent.
			TransactionCoordinator tc = session.getTransactionCoordinator();
			if (tc != null && tc.isJoined()) {
				tc.getLocalSynchronizations().registerSynchronization(new PostTransactionQueryUpdate(tc, session, key, value));
				// no need to synchronize as the transaction will be accessed by only one thread
				Map map = transactionContext.get(session);
				if (map == null) {
					transactionContext.put(session, map = new HashMap());
				}
				map.put(key, value);
				return;
			}
			// Here we don't want to suspend the tx. If we do:
			// 1) We might be caching query results that reflect uncommitted
			// changes. No tx == no WL on cache node, so other threads
			// can prematurely see those query results
			// 2) No tx == immediate replication. More overhead, plus we
			// spread issue #1 above around the cluster

			// Add a zero (or quite low) timeout option so we don't block.
			// Ignore any TimeoutException. Basically we forego caching the
			// query result in order to avoid blocking.
			// Reads are done with suspended tx, so they should not hold the
			// lock for long.  Not caching the query result is OK, since
			// any subsequent read will just see the old result with its
			// out-of-date timestamp; that result will be discarded and the
			// db query performed again.
			putCache.put( key, value );
		}
	}

	private class PostTransactionQueryUpdate implements Synchronization {
		private final TransactionCoordinator tc;
		private final SessionImplementor session;
		private final Object key;
		private final Object value;

		public PostTransactionQueryUpdate(TransactionCoordinator tc, SessionImplementor session, Object key, Object value) {
			this.tc = tc;
			this.session = session;
			this.key = key;
			this.value = value;
		}

		@Override
		public void beforeCompletion() {
		}

		@Override
		public void afterCompletion(int status) {
			transactionContext.remove(session);
			switch (status) {
				case Status.STATUS_COMMITTING:
				case Status.STATUS_COMMITTED:
					try {
						// TODO: isolation without obtaining Connection
						tc.createIsolationDelegate().delegateWork(new WorkExecutorVisitable<Void>() {
							@Override
							public Void accept(WorkExecutor<Void> executor, Connection connection) throws SQLException {
								putCache.put(key, value);
								return null;
							}
						}
						, putCacheRequiresTransaction);
					}
					catch (HibernateException e) {
						// silently fail any exceptions
						if (log.isTraceEnabled()) {
							log.trace("Exception during query cache update", e);
						}
					}
					break;
				default:
					// it would be nicer to react only on ROLLING_BACK and ROLLED_BACK statuses
					// but TransactionCoordinator gives us UNKNOWN on rollback
					break;
			}
		}
	}
}
