/*
 * Hibernate OGM, Domain model persistence for NoSQL datastores
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.ogm.datastore.infinispan.persistencestrategy.kind.externalizer.impl;

import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Collections;
import java.util.Set;

import org.hibernate.ogm.datastore.infinispan.InfinispanDialect;
import org.hibernate.ogm.datastore.infinispan.impl.InfinispanEmbeddedDatastoreProvider;
import org.hibernate.ogm.datastore.infinispan.persistencestrategy.common.externalizer.impl.ExternalizerIds;
import org.hibernate.ogm.datastore.infinispan.persistencestrategy.common.externalizer.impl.VersionChecker;
import org.hibernate.ogm.model.impl.DefaultIdSourceKeyMetadata;
import org.hibernate.ogm.model.key.spi.IdSourceKey;
import org.hibernate.ogm.model.key.spi.IdSourceKeyMetadata;
import org.infinispan.commons.marshall.AdvancedExternalizer;

/**
 * An externalizer for serializing and de-serializing {@link IdSourceKey} instances. Implicitly used by
 * {@link InfinispanDialect} which stores keys as is in the Infinispan data store.
 * <p>
 * This externalizer is automatically registered with the cache manager when starting the
 * {@link InfinispanEmbeddedDatastoreProvider}, so it's not required to configure the externalizer in the Infinispan
 * configuration file.
 *
 * @author Gunnar Morling
 */
// As an implementation of AdvancedExternalizer this is never serialized according to the Externalizer docs
@SuppressWarnings("serial")
public class IdSourceKeyExternalizer implements AdvancedExternalizer<IdSourceKey> {

	public static final IdSourceKeyExternalizer INSTANCE = new IdSourceKeyExternalizer();

	/**
	 * Format version of the key type; allows to apply version dependent deserialization logic in the future if
	 * required; to be incremented when adding new fields to the serialized structure
	 */
	private static final int VERSION = 1;

	private static final Set<Class<? extends IdSourceKey>> TYPE_CLASSES = Collections.<Class<? extends IdSourceKey>>singleton( IdSourceKey.class );

	private IdSourceKeyExternalizer() {
	}

	@Override
	public void writeObject(ObjectOutput output, IdSourceKey key) throws IOException {
		output.writeInt( VERSION );
		output.writeUTF( key.getTable() );
		//Wrapping in String[] and Object[] respectively as this used to be the format,
		//to maintain compatibility with Hibernate OGM 5.0
		output.writeObject( new String[] { key.getColumnName() } );
		output.writeObject( new Object[] { key.getColumnValue() } );
	}

	@Override
	public IdSourceKey readObject(ObjectInput input) throws IOException, ClassNotFoundException {
		VersionChecker.readAndCheckVersion( input, VERSION, IdSourceKey.class );

		String tableName = input.readUTF();
		String[] columnNames = (String[]) input.readObject();
		Object[] values = (Object[]) input.readObject();

		IdSourceKeyMetadata metadata = DefaultIdSourceKeyMetadata.forTable( tableName, columnNames[0], null );
		return IdSourceKey.forTable( metadata , (String) values[0] );
	}

	@Override
	public Set<Class<? extends IdSourceKey>> getTypeClasses() {
		return TYPE_CLASSES;
	}

	@Override
	public Integer getId() {
		return ExternalizerIds.PER_KIND_ID_GENERATOR_KEY;
	}
}
