/*
 * Hibernate Tools, Tooling for your Hibernate Projects
 *
 * Copyright 2023-2025 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.hibernate.tool.language;

import org.hibernate.SharedSessionContract;
import org.hibernate.query.SelectionQuery;

/**
 * Hibernate Assistant allows interacting with an underlying LLM to help you retrieve persistent data.
 * It leverages Hibernate ORM's mapping models, query language, cross-platform support and
 * built-in data restrictions to make access to information stored in relational databases
 * as easy as a natural language prompt.
 */
public interface HibernateAssistant {
	/**
	 * Creates a {@link SelectionQuery} by providing the specified natural language {@code message} to the LLM
	 * and interpreting the obtained response.
	 *
	 * @param message the natural language prompt
	 * @param session Hibernate session
	 *
	 * @return the {@link SelectionQuery} generated by the LLM
	 */
	default SelectionQuery<?> createAiQuery(String message, SharedSessionContract session) {
		return createAiQuery( message, session, null );
	}

	/**
	 * Creates a {@link SelectionQuery} by providing the specified natural language {@code message} to the LLM
	 * and interpreting the obtained response.
	 *
	 * @param message the natural language prompt
	 * @param session Hibernate session
	 * @param resultType The {@link Class} representing the expected query result type
	 *
	 * @return the {@link SelectionQuery} generated by the LLM
	 */
	<T> SelectionQuery<T> createAiQuery(String message, SharedSessionContract session, Class<T> resultType);

	/**
	 * Prompts the underlying LLM with the provided natural language message and tries to answer it with
	 * data extracted from the database through the persistence model.
	 *
	 * @param message the natural language request
	 * @param session Hibernate session
	 *
	 * @return a natural language response based on the results of the query
	 */
	String executeQuery(String message, SharedSessionContract session);

	/**
	 * Executes the given {@link SelectionQuery}, and provides a natural language
	 * response by passing the resulting data back to the underlying LLM.
	 * <p>
	 * To directly obtain a natural language response from a natural language prompt,
	 * you can use {@link #executeQuery(String, SharedSessionContract)} instead.
	 * <p>
	 * If you wish to execute the query manually and obtain the structured results yourself,
	 * you should use {@link SelectionQuery}'s direct execution methods, e.g. {@link SelectionQuery#getResultList()}
	 * or {@link SelectionQuery#getSingleResult()}.
	 *
	 * @param query the AI query to execute
	 * @param session the session in which to execute the query
	 *
	 * @return a natural language response based on the results of the query
	 */
	String executeQuery(SelectionQuery<?> query, SharedSessionContract session);

	/**
	 * Reset the assistant's current chat context. This can be helpful when
	 * creating a new {@link SelectionQuery} that should not rely on the context
	 * of previous requests.
	 */
	void clear();
}
