/*
 * Copyright 2010 Red Hat, Inc.
 * Red Hat licenses this file to you under the Apache License, version
 * 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *    http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.  See the License for the specific language governing
 * permissions and limitations under the License.
 */
package org.hornetq.core.protocol.stomp;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;

/**
 *
 * @author <a href="mailto:hgao@redhat.com">Howard Gao</a>
 */
public class HornetQStompException extends Exception
{
   public static final int NONE = 0;
   public static final int INVALID_EOL_V10 = 1;
   public static final int INVALID_COMMAND = 2;

   private static final long serialVersionUID = -274452327574950068L;

   private int code = NONE;
   private final List<Header> headers = new ArrayList<Header>(10);
   private String body;
   private final VersionedStompFrameHandler handler;
   private boolean disconnect;

   public HornetQStompException(StompConnection connection, String msg)
   {
      super(msg);
      handler = connection.getFrameHandler();
   }

   public HornetQStompException(String msg)
   {
      super(msg);
      handler = null;
   }

   public HornetQStompException(String msg, Throwable t)
   {
      super(msg, t);
      this.body = t.getMessage();
      handler = null;
   }

   //used for version control logic
   public HornetQStompException(int code, String details)
   {
      super(details);
      this.code = code;
      this.body = details;
      handler = null;
   }

   void addHeader(String header, String value)
   {
      headers.add(new Header(header, value));
   }

   public void setBody(String body)
   {
      this.body = body;
   }

   public StompFrame getFrame()
   {
      StompFrame frame = null;
      if (handler == null)
      {
         frame = new StompFrame("ERROR");
      }
      else
      {
         frame = handler.createStompFrame("ERROR");
      }
      frame.addHeader("message", this.getMessage());
      for (Header header : headers)
      {
        frame.addHeader(header.key, header.val);
      }

      if (body != null)
      {
         try
         {
            frame.setByteBody(body.getBytes("UTF-8"));
         }
         catch (UnsupportedEncodingException e)
         {
         }
      }
      else
      {
         frame.setByteBody(new byte[0]);
      }
     frame.setNeedsDisconnect(disconnect);
      return frame;
   }

   private static final class Header
   {
      public final String key;
      public final String val;

      public Header(String key, String val)
      {
         this.key = key;
         this.val = val;
      }
   }

   public void setDisconnect(boolean b)
   {
      disconnect = b;
   }
   
   public int getCode()
   {
      return code;
   }
}
