/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.persistence.cassandra;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.Metadata;
import com.datastax.driver.core.PoolingOptions;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.QueryOptions;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.concurrent.Executor;
import org.infinispan.commons.configuration.ConfiguredBy;
import org.infinispan.commons.io.ByteBuffer;
import org.infinispan.commons.io.ByteBufferFactory;
import org.infinispan.commons.marshall.StreamingMarshaller;
import org.infinispan.filter.KeyFilter;
import org.infinispan.marshall.core.MarshalledEntry;
import org.infinispan.marshall.core.MarshalledEntryFactory;
import org.infinispan.persistence.TaskContextImpl;
import org.infinispan.persistence.cassandra.configuration.CassandraStoreConfiguration;
import org.infinispan.persistence.cassandra.configuration.CassandraStoreConnectionPoolConfiguration;
import org.infinispan.persistence.cassandra.configuration.CassandraStoreServerConfiguration;
import org.infinispan.persistence.cassandra.logging.Log;
import org.infinispan.persistence.spi.AdvancedCacheLoader;
import org.infinispan.persistence.spi.AdvancedCacheWriter;
import org.infinispan.persistence.spi.AdvancedLoadWriteStore;
import org.infinispan.persistence.spi.InitializationContext;
import org.infinispan.persistence.spi.PersistenceException;
import org.infinispan.util.TimeService;
import org.infinispan.util.logging.LogFactory;

@ConfiguredBy(value=CassandraStore.class)
public class CassandraStore
implements AdvancedLoadWriteStore {
    private static final Log log = (Log)LogFactory.getLog(CassandraStore.class, Log.class);
    private static final boolean trace = log.isTraceEnabled();
    private InitializationContext ctx;
    private CassandraStoreConfiguration configuration;
    private Cluster cluster;
    private Session session;
    private TimeService timeService;
    private StreamingMarshaller marshaller;
    private MarshalledEntryFactory marshalledEntryFactory;
    private ByteBufferFactory byteBufferFactory;
    private String entryTable;
    private PreparedStatement writeStatement;
    private PreparedStatement selectStatement;
    private PreparedStatement containsStatement;
    private PreparedStatement selectAllStatement;
    private PreparedStatement deleteStatement;
    private PreparedStatement sizeStatement;
    private PreparedStatement clearStatement;

    public void init(InitializationContext initializationContext) {
        this.ctx = initializationContext;
        this.timeService = this.ctx.getTimeService();
        this.marshaller = this.ctx.getMarshaller();
        this.marshalledEntryFactory = this.ctx.getMarshalledEntryFactory();
        this.byteBufferFactory = this.ctx.getByteBufferFactory();
        this.configuration = (CassandraStoreConfiguration)this.ctx.getConfiguration();
    }

    public void start() {
        try {
            PoolingOptions poolingOptions = new PoolingOptions();
            CassandraStoreConnectionPoolConfiguration poolConfig = this.configuration.connectionPool();
            poolingOptions.setPoolTimeoutMillis(poolConfig.poolTimeoutMillis());
            poolingOptions.setHeartbeatIntervalSeconds(poolConfig.heartbeatIntervalSeconds());
            poolingOptions.setIdleTimeoutSeconds(poolConfig.idleTimeoutSeconds());
            QueryOptions queryOptions = new QueryOptions();
            queryOptions.setConsistencyLevel(this.configuration.consistencyLevel());
            queryOptions.setSerialConsistencyLevel(this.configuration.serialCconsistencyLevel());
            Cluster.Builder builder = Cluster.builder();
            builder.withPoolingOptions(poolingOptions);
            builder.withQueryOptions(queryOptions);
            ArrayList<InetSocketAddress> servers = new ArrayList<InetSocketAddress>();
            for (CassandraStoreServerConfiguration cassandraStoreServerConfiguration : this.configuration.servers()) {
                InetAddress host = InetAddress.getByName(cassandraStoreServerConfiguration.host());
                int port = cassandraStoreServerConfiguration.port();
                servers.add(new InetSocketAddress(host, port));
            }
            builder.addContactPointsWithPorts(servers);
            this.cluster = builder.build();
            if (this.configuration.autoCreateKeyspace().booleanValue()) {
                this.createKeySpace();
            }
            this.session = this.cluster.connect(this.configuration.keyspace());
            this.entryTable = this.configuration.entryTable();
            this.writeStatement = this.session.prepare("INSERT INTO " + this.entryTable + " (key, value, metadata) VALUES (?, ?, ?) USING TTL ?");
            this.deleteStatement = this.session.prepare("DELETE FROM " + this.entryTable + " WHERE key=?");
            this.selectStatement = this.session.prepare("SELECT value, metadata FROM " + this.entryTable + " WHERE key=?");
            this.containsStatement = this.session.prepare("SELECT key FROM " + this.entryTable + " WHERE key=?");
            this.selectAllStatement = this.session.prepare("SELECT key, value, metadata FROM " + this.entryTable);
            this.sizeStatement = this.session.prepare("SELECT count(*) FROM " + this.entryTable);
            this.clearStatement = this.session.prepare("TRUNCATE " + this.entryTable);
        }
        catch (Exception e) {
            throw log.errorCommunicating(e);
        }
        this.entryTable = this.configuration.entryTable();
        log.debug("Cassandra cache store started.");
    }

    private void createKeySpace() {
        try (Session session = this.cluster.connect();){
            boolean entryTableExists;
            boolean keyspaceExists;
            Metadata clusterMetadata = this.cluster.getMetadata();
            boolean bl = keyspaceExists = clusterMetadata.getKeyspace(this.configuration.keyspace()) != null;
            if (!keyspaceExists) {
                log.debug("Creating a keyspace " + this.configuration.keyspace());
                session.execute("CREATE KEYSPACE IF NOT EXISTS " + this.configuration.keyspace() + " WITH replication " + "= {'class':'SimpleStrategy', 'replication_factor':1};");
            }
            boolean bl2 = entryTableExists = clusterMetadata.getKeyspace(this.configuration.keyspace()).getTable(this.configuration.entryTable()) != null;
            if (!entryTableExists) {
                log.debug("Creating an entry table " + this.configuration.entryTable());
                session.execute("CREATE TABLE " + this.configuration.keyspace() + "." + this.configuration.entryTable() + " (" + "key blob PRIMARY KEY," + "value blob," + "metadata blob);");
            }
        }
        catch (Exception e) {
            throw log.errorCreatingKeyspace(e);
        }
    }

    public void write(MarshalledEntry marshalledEntry) {
        if (trace) {
            log.tracef("Writing to Cassandra: %s", marshalledEntry);
        }
        int ttl = 0;
        java.nio.ByteBuffer metadata = null;
        if (marshalledEntry.getMetadata() != null && marshalledEntry.getMetadata().expiryTime() > -1L) {
            long now = this.timeService.wallClockTime();
            long expireAt = marshalledEntry.getMetadata().expiryTime();
            ttl = (int)(expireAt - now) / 1000;
            metadata = java.nio.ByteBuffer.wrap(marshalledEntry.getMetadataBytes().getBuf());
        }
        java.nio.ByteBuffer key = java.nio.ByteBuffer.wrap(marshalledEntry.getKeyBytes().getBuf());
        java.nio.ByteBuffer value = java.nio.ByteBuffer.wrap(marshalledEntry.getValueBytes().getBuf());
        try {
            this.session.execute((Statement)this.writeStatement.bind(new Object[]{key, value, metadata, ttl}));
            if (trace) {
                log.tracef("Stored: %s", marshalledEntry);
            }
        }
        catch (Exception e) {
            throw log.errorWritingEntry(e);
        }
    }

    public boolean delete(Object o) {
        if (trace) {
            log.tracef("Deleting from Cassandra: %s", o);
        }
        if (this.contains(o)) {
            try {
                this.session.execute((Statement)this.deleteStatement.bind(new Object[]{this.marshall(o)}));
            }
            catch (Exception e) {
                throw log.errorDeletingEntry(e);
            }
            if (trace) {
                log.tracef("Deleted: %s", o);
            }
            return true;
        }
        return false;
    }

    public MarshalledEntry load(Object o) {
        Row row;
        if (trace) {
            log.tracef("Loading from Cassandra: %s", o);
        }
        try {
            row = this.session.execute((Statement)this.selectStatement.bind(new Object[]{this.marshall(o)})).one();
        }
        catch (Exception e) {
            throw log.errorLoadingEntry(e);
        }
        if (row == null) {
            return null;
        }
        byte[] valueBytes = row.getBytes(0).array();
        ByteBuffer valueBuffer = this.byteBufferFactory.newByteBuffer(valueBytes, 0, valueBytes.length);
        ByteBuffer metadataBuffer = null;
        if (row.getBytes(1) != null) {
            byte[] metadataBytes = row.getBytes(1).array();
            metadataBuffer = this.byteBufferFactory.newByteBuffer(metadataBytes, 0, metadataBytes.length);
        }
        MarshalledEntry marshalledEntry = this.marshalledEntryFactory.newMarshalledEntry(o, valueBuffer, metadataBuffer);
        if (trace) {
            log.tracef("Loaded: %s", marshalledEntry);
        }
        return marshalledEntry;
    }

    public boolean contains(Object o) {
        boolean contains;
        if (trace) {
            log.tracef("Cassandra contains? key: %s", o);
        }
        try {
            boolean bl = contains = this.session.execute((Statement)this.containsStatement.bind(new Object[]{this.marshall(o)})).one() != null;
            if (contains) {
                if (trace) {
                    log.tracef("Cassandra contains: %s", o);
                }
                return true;
            }
        }
        catch (Exception e) {
            throw log.errorCommunicating(e);
        }
        return contains;
    }

    public void stop() {
        this.cluster.close();
    }

    public void process(KeyFilter filter, AdvancedCacheLoader.CacheLoaderTask task, Executor executor, boolean fetchValue, boolean fetchMetadata) {
        TaskContextImpl taskContext = new TaskContextImpl();
        ResultSet rows = null;
        try {
            rows = this.session.execute((Statement)this.selectAllStatement.bind());
        }
        catch (Exception e) {
            throw log.errorCommunicating(e);
        }
        for (Row row : rows) {
            if (taskContext.isStopped()) break;
            byte[] keyBytes = row.getBytes(0).array();
            Object key = this.unmarshall(keyBytes);
            if (filter != null && !filter.accept(key)) continue;
            try {
                MarshalledEntry marshalledEntry;
                byte[] valueBytes = row.getBytes(1).array();
                byte[] metadataBytes = row.getBytes(2) != null ? row.getBytes(2).array() : null;
                ByteBuffer keyBuffer = this.byteBufferFactory.newByteBuffer(keyBytes, 0, keyBytes.length);
                ByteBuffer valueBuffer = this.byteBufferFactory.newByteBuffer(valueBytes, 0, valueBytes.length);
                ByteBuffer metadataBuffer = null;
                if (metadataBytes != null) {
                    metadataBuffer = this.byteBufferFactory.newByteBuffer(metadataBytes, 0, metadataBytes.length);
                }
                if ((marshalledEntry = this.marshalledEntryFactory.newMarshalledEntry(keyBuffer, valueBuffer, metadataBuffer)) == null) continue;
                task.processEntry(marshalledEntry, (AdvancedCacheLoader.TaskContext)taskContext);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return;
            }
        }
    }

    public int size() {
        int size = 0;
        try {
            size = (int)this.session.execute((Statement)this.sizeStatement.bind()).one().getLong(0);
        }
        catch (Exception e) {
            throw log.errorCommunicating(e);
        }
        if (trace) {
            log.tracef("Size of Cassandra store: %d", size);
        }
        return size;
    }

    public void clear() {
        try {
            if (trace) {
                log.trace("Clearing Cassandra store");
            }
            this.session.execute((Statement)this.clearStatement.bind());
            if (trace) {
                log.trace("Cleared Cassandra store");
            }
        }
        catch (Exception e) {
            throw log.errorClearing(e);
        }
    }

    public void purge(Executor threadPool, AdvancedCacheWriter.PurgeListener listener) {
    }

    private java.nio.ByteBuffer marshall(Object o) {
        try {
            return java.nio.ByteBuffer.wrap(this.marshaller.objectToByteBuffer(o));
        }
        catch (Exception e) {
            throw new PersistenceException((Throwable)e);
        }
    }

    private Object unmarshall(byte[] bytes) {
        try {
            return this.marshaller.objectFromByteBuffer(bytes);
        }
        catch (Exception e) {
            throw new PersistenceException((Throwable)e);
        }
    }
}

