package org.jboss.hal.testsuite.util;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.nio.file.Path;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import org.apache.commons.io.FileUtils;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.exporter.ZipExporter;
import org.wildfly.extras.creaper.core.online.OnlineManagementClient;

/**
 * Utility class for creating/removing (customer) static modules.
 */
public class ModuleUtils {

    private final Path modulesPath;

    public ModuleUtils(OnlineManagementClient client) throws IOException {
        this.modulesPath = new PathOperations(client).getModulesPath();
    }

    /**
     * Create new static module with main slot and module name generated by joining {@code modulePath} elements with dot.
     * @param modulePath if the path is absolute use it, otherwise resolve it as a relative path to the modules
     * directory
     * @param dependencyModuleNames names of modules on which module to be created depends
     * @return module name
     * @throws IllegalArgumentException if some module already exists on {@code modulePath}
     */
    public String createModule(Path modulePath, Archive<?> resourceArchive,
            String... dependencyModuleNames) throws IOException {
        String moduleName = getModuleName(modulePath);
        File moduleSlotDir = createModuleSlotDir(modulePath, "main");
        createResourceArchive(moduleSlotDir, resourceArchive);
        createModuleDescriptor(moduleName, moduleSlotDir, resourceArchive, dependencyModuleNames);
        return moduleName;
    }

    /**
     * Removes the module directory
     * @param modulePath if the path is absolute use it, otherwise resolve it as a relative path to the modules
     * directory
     * @throws IOException
     */
    public void removeModule(Path modulePath) throws IOException {
        FileUtils.forceDelete(modulesPath.resolve(modulePath).toFile());
    }

    private String getModuleName(Path modulePath) {
        Iterable<Path> modulePathIterable = () -> modulePath.iterator();
        return StreamSupport.stream(modulePathIterable.spliterator(), false).map(pathElement -> pathElement.toString())
                .collect(Collectors.joining("."));
    }

    private File createModuleSlotDir(Path modulePath, String slot) throws IOException {
        Path absoluteModulePath = modulesPath.resolve(modulePath);
        File moduleSlotDir = absoluteModulePath.resolve(slot).toFile();
        if (moduleSlotDir.exists()) {
            throw new IllegalArgumentException("Some module already exists on path '" + absoluteModulePath + "'.");
        }
        FileUtils.forceMkdir(moduleSlotDir);
        return moduleSlotDir;
    }

    private void createResourceArchive(File moduleSlotDir, Archive<?> resourceArchive) {
        resourceArchive.as(ZipExporter.class).exportTo(new File(moduleSlotDir, resourceArchive.getName()));
    }

    private void createModuleDescriptor(String moduleName, File moduleSlotDir, Archive<?> resourceArchive,
            String... dependencyModuleNames) throws FileNotFoundException, UnsupportedEncodingException {
        try (PrintWriter wr = new PrintWriter(new File(moduleSlotDir, "module.xml"), "UTF-8")) {
            wr.println("<?xml version='1.0' encoding='UTF-8'?>");
            wr.format("<module xmlns=\"urn:jboss:module:1.1\" name=\"%s\">", moduleName);
            wr.println();
            wr.println("<resources>");
            wr.format("<resource-root path=\"%s\"/>", resourceArchive.getName());
            wr.println();
            wr.println("</resources>");
            wr.println("<dependencies>");
            for (String dependencyModule : dependencyModuleNames) {
                wr.format("<module name=\"%s\"/>", dependencyModule);
                wr.println();
            }
            wr.println("</dependencies>");
            wr.println("</module>");
        }
    }
}
