/*
 * Copyright 2012 JBoss, by Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.as.websockets;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.io.IOException;

/**
 * Represents a handle to a single WebSocket connection. It has reader and writer methods to get data in and out.
 *
 * TODO: Implement support for binary frames.
 *
 * @author Mike Brock
 */
public interface WebSocket {

  /**
   * A unique ID associated with the socket, which can be used for session association. This ID is generated by the
   * WebSockets framework as a random hash when the socket is open and has no association with any external API or
   * the websocket handshake process.
   *
   * @return A hex string representing the unique ID of the socket.
   */
  public String getSocketID();

  /**
   * Write an text frame to the websocket. All String data will be UTF-8 encoded on the wire.
   *
   * @param text the UTF-8 text string
   * @throws IOException
   */
  public void writeTextFrame(String text) throws IOException;

  /**
   * Read a single text frame -- if available.
   *
   * @return the UTF-8 text payload string.
   * @throws IOException
   */
  public String readTextFrame() throws IOException;


  /**
   * Return the HTTP Session with which this WebSocket is associated.
   * @return an instance of the HttpSession
   */
  public HttpSession getHttpSession();


  /**
   * Get an instance of the initial ServletRequest which was responsible for opening this WebSocket. Note: that
   * this object remains the same for the duration of the WebSocket session. There is <strong>no</strong> unique
   * request associated with individual websocket frames.
   *
   * @return an instance of the HttpServletRequest which opened this WebSocket.
   */
  public HttpServletRequest getServletRequest();

  /**
   * Terminates the connection with the client and closes the socket.
   */
  public void closeSocket() throws IOException;
}
