/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.classpool.base;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javassist.CtClass;

import org.jboss.classpool.domain.AbstractClassPoolDomain;
import org.jboss.classpool.domain.ClassPoolDomain;
import org.jboss.classpool.helpers.ClassLoaderUtils;
import org.jboss.classpool.plugins.DelegatingClassPool;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 100482 $
 */
public class BaseClassPoolDomain extends AbstractClassPoolDomain
{
   private final String domainName;
   
   private final List<DelegatingClassPool> delegatingPools = new CopyOnWriteArrayList<DelegatingClassPool>();
   
   private final ParentDelegationStrategy parentDelegationStrategy;
   
   private final ReadWriteLock lock = new ReentrantReadWriteLock();
   
   private volatile int modCount;
   
   private volatile int parentModCount;
   
   public BaseClassPoolDomain(String domainName, ClassPoolDomain parent, boolean parentFirst)
   {
      this(domainName, 
            new DefaultParentDelegationStrategy(
                  parent, 
                  parentFirst, 
                  DefaultClassPoolToClassPoolDomainAdaptorFactory.getInstance())
      );
   }
   
   protected BaseClassPoolDomain(String domainName, ParentDelegationStrategy parentDelegationStrategy)
   {
      this.domainName = domainName;
      this.parentDelegationStrategy = parentDelegationStrategy;
      parentDelegationStrategy.setDomain(this);
      this.parentModCount = parentDelegationStrategy.getParentModCount();
      if (logger.isTraceEnabled()) logger.trace("Created " + this + " parentDelegationStrategy:" + parentDelegationStrategy);
   }
   
   public void addClassPool(DelegatingClassPool pool)
   {
      lockWrite();
      try
      {
         if (!delegatingPools.contains(pool))
         {
            if (logger.isTraceEnabled()) logger.trace(this + " adding pool " + pool);
            delegatingPools.add(pool);
            parentModCount = parentDelegationStrategy.getParentModCount();
            modCount++;
         }
      }
      finally
      {
         unlockWrite();
      }
   }
   
   public void removeClassPool(DelegatingClassPool pool)
   {
      lockWrite();
      try
      {
         if (logger.isTraceEnabled()) logger.trace(this + " removing pool " + pool);
         if (delegatingPools.remove(pool))
         {
            parentModCount = parentDelegationStrategy.getParentModCount();
            modCount++;
         }
      }
      finally
      {
         unlockWrite();
      }
   }

   public int getModCount()
   {
      //Check if the parent mod count has changed (pool was added/removed in parent)
      if (parentModCount < parentDelegationStrategy.getParentModCount())
      {
         lockWrite();
         try
         {
            int parentCount = parentDelegationStrategy.getParentModCount();
            if (parentCount > parentModCount)
            {
               this.parentModCount = parentCount; 
               return ++modCount;
            }
         }
         finally
         {
            unlockWrite();
         }
      }
      
      return modCount;
   }


   public CtClass getCachedOrCreate(DelegatingClassPool initiatingPool, String classname, boolean create)
   {
      boolean trace = logger.isTraceEnabled();
      
      String resourceName = ClassLoaderUtils.getResourceName(classname);
      
      CtClass clazz = getCachedOrCreate(initiatingPool, classname, resourceName, create, trace);
      
      if (clazz == null)
      {
         clazz = getCachedOrCreateFromPoolParent(initiatingPool, classname, create, trace);
      }
      
      return clazz;
   }
   
   public CtClass getCachedOrCreate(DelegatingClassPool initiatingPool, String classname, String resourceName, boolean create, boolean trace)
   {
      if (trace) logger.trace(this + " looking for " + classname);
         
      CtClass clazz = null;
      if (isParentBefore(classname))
      {
         if (trace) logger.trace(this + " checking parent first for " + classname);
         clazz = getCachedOrCreateFromParent(initiatingPool, classname, resourceName, create, trace);
      }
      if (clazz == null)
      {
         List<DelegatingClassPool> pools = getPoolsForClassName(classname);
         if (pools.size() > 0)
         {
            for (DelegatingClassPool pool : pools)
            {
               clazz = pool.loadLocally(classname, resourceName, create);
               if (clazz != null)
               {
                  break;
               }
            }
         }
      }
      if (clazz == null && isParentAfter(classname))
      {
         if (trace) logger.trace(this + " checking parent last for " + classname);
         clazz = getCachedOrCreateFromParent(initiatingPool, classname, resourceName, create, trace);
      }
      if (trace) logger.trace(this + " found " + classname + " in " + (clazz == null ? "null" : clazz.getClassPool()));
      return clazz;
   }

   @Override
   protected CtClass getCachedOrCreateFromParent(DelegatingClassPool initiatingPool, String classname, String resourceName, boolean create, boolean trace)
   {
      return parentDelegationStrategy.getCachedOrCreateFromParent(initiatingPool, classname, resourceName, create, trace);
   }
   
   public String toString()
   {
      return "[" + super.toString() + " name:" + domainName + "]";
   }

   protected boolean isParentBefore(String classname)
   {
      return parentDelegationStrategy.isParentBefore(classname);
   }
   
   protected boolean isParentAfter(String classname)
   {
      return parentDelegationStrategy.isParentAfter(classname);
   }
   
   protected List<DelegatingClassPool> getPoolsForClassName(String classname)
   {
      return delegatingPools;
   }
   
   protected void lockRead()
   {
      lock.readLock().lock();
   }
   
   protected void unlockRead()
   {
      lock.readLock().unlock();
   }
   
   protected void lockWrite()
   {
      lock.writeLock().lock();
   }
   
   protected void unlockWrite()
   {
      lock.writeLock().unlock();
   }

}
