/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.classpool.plugins;

import java.util.Set;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.NotFoundException;

import org.jboss.classpool.base.BaseClassPool;
import org.jboss.classpool.scoped.ScopedClassPoolRepository;
import org.jboss.classpool.spi.AbstractClassPool;
import org.jboss.util.collection.ConcurrentSet;


/**
 * ClassPool for class loaders not backed by a repository/classloading domain
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 104808 $
 */
public class NonDelegatingClassPool extends BaseClassPool
{
   private Set<String> parentBlacklist = new ConcurrentSet<String>(100, .75f, 3);
   
   private Set<String> parentWhitelist = new ConcurrentSet<String>(100, .75f, 3);
   
   public NonDelegatingClassPool(ClassLoader cl, ClassPool src, ScopedClassPoolRepository repository, boolean parentFirst)
   {
      super(cl, src, repository, AbstractClassPool.SEARCH_LOCAL_ONLY_STRATEGY);
      super.childFirstLookup = !parentFirst;
   }

   @Override
   public CtClass createCtClass(String classname, boolean useCache)
   {
      boolean trace = logger.isTraceEnabled();
      CtClass clazz = null;
      if (!childFirstLookup)
      {
         if (trace)logger.trace(this + " attempting to create " + classname + " in parent pool (parentFirst)");
         clazz = createParentCtClass(classname, useCache, trace);
      }
      if (clazz == null && isLocalResource(getResourceName(classname), trace))
      {
         clazz = super.createCtClass(classname, useCache);
      }
      if (childFirstLookup && clazz == null)
      {
         if (trace)logger.trace(this + " attempting to create " + classname + " in parent pool (parentLast)");
         clazz = createParentCtClass(classname, useCache, trace);
      }
      
      if (trace)logger.trace(this + " created " + classname + " " + getClassPoolLogStringForClass(clazz));
      return clazz;
   }
   
   /**
    * Uses the parent ClassPool, if available, to create the CtClass representing
    * {@code classname}.
    * 
    * @param classname the name of the queried CtClass
    * @param useCache  whether the ScopedClassPool cache should be used to search for the CtClass
    * @param trace     enables tracing
    * @return          a {@code CtClass} representing {@code classname}
    */
   private CtClass createParentCtClass(String classname, boolean useCache, boolean trace)
   {
      if (parent == null)
      {
         return null;
      }

      //Make parent create class
      if (parent instanceof BaseClassPool)
      {
         return ((BaseClassPool)parent).createCtClass(classname, useCache);
      }
      else
      {
         return plainParentGet(classname);
      }
   }
   
   private CtClass plainParentGet(String classname)
   {
      try
      {
         if (parentWhitelist.contains(classname))
            return parent.get(classname);
      }
      catch (NotFoundException e)
      {
         //This should not happen
         logger.warn(classname + " not loaded from parent despite having been loaded before");
         return null;
      }
      
      if (parentBlacklist.contains(classname))
         return null;

      try
      {
         CtClass clazz = parent.get(classname);
         parentWhitelist.add(classname);
         return clazz;
      }
      catch (NotFoundException e)
      {
         parentBlacklist.add(classname);
      }
      
      return null;
   }


}
