/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ha.framework.interfaces;

import java.io.Serializable;
import java.util.List;
import java.util.Collection;

/** 
 *
 * DistributedReplicantManager is a service on top of {@link HAPartition} that provides a
 * cluster-wide distributed state which is node-specific. The DistributedReplicantManager (DRM) 
 * service provides a &lt;String key, Serializable replicant&gt; tuple map.  Each node in the cluster
 * stores its own data for a given key.  This data is replicated so that reading values for a key
 * is always performed locally.  
 * <p>
 * Typical usages for DistributedReplicantManager are:
 * <ol>
 * <li>To serve as a distributed registry for remote service endpoints for 
 * clustered services. In this usage, the Serializable <code>replicants</code> 
 * that services store in the DRM are information (e.g. RMI stubs, JBoss Remoting 
 * <code>InvokerLocator</code> instances, HTTP URLs etc) that remote clients 
 * would need to contact the clustered service.  This usage of DRM drives
 * the smart clustered proxy implementations used by JBoss AS.</li>
 * <li>To simply serve as a distributed registry showing on what cluster
 * nodes particular services are deployed. Here the Serializable 
 * <code>replicants</code> that services store in the DRM are usually
 * just simple placeholder strings; the presence of the tuple itself is
 * what indicates the service is available on a particular node. This usage
 * of DRM drives JBoss AS services like HASingleton.</li>
 * </ol>
 * 
 *   @author  <a href="mailto:bill@burkecentral.com">Bill Burke</a>.
 *   @author  <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>.
 *   @author  <a href="mailto:brian.stansberry@jboss.com">Brian Stansberry</a>
 *   @author  <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 *   @version $Revision: 69398 $
 *
 * <p><b>Revisions:</b><br>
 * <p><b>2001/10/31: marcf</b>
 * <ol>
 *   <li>DRM is no longer remote
 * </ol>
 * <p><b>2002/08/23: Sacha Labourey</b>
 * <ol>
 *   <li>added isMasterReplica
 * </ol>
 */
public interface DistributedReplicantManager
{
   /**
    * When a particular key in the DistributedReplicantManager table gets modified, all listeners
    * will be notified of replicant changes for that key.
    */
   public interface ReplicantListener
   {
      /**
       * Callback called when the content/list of replicant for a given replicant key has changed
       * @param key The name of the key of the replicant that has changed
       * @param newReplicants The list of new replicants for the give replicant key.
       *                      This list will be in a consistent order on all
       *                      cluster nodes on which the current viewId is
       *                      in effect
       * @param newReplicantsViewId The new replicant view id corresponding to this change
       * @param merge         Is the change due to a merge (i.e. the healing of a 
       *                      cluster split)?
       */      
      public void replicantsChanged(String key, List newReplicants, int newReplicantsViewId, boolean merge);
   }

   /**
    * Subscribe a new {@link ReplicantListener listener } for replicant changes
    * @param key Name of the replicant, must be identical cluster-wide for all related replicants
    * @param subscriber The subscribing {@link ReplicantListener}
    */   
   public void registerListener(String key, ReplicantListener subscriber);
   /**
    * Unsubscribe a {@link ReplicantListener listener} that had subscribed for replicant changes
    * @param key Name of the replicant, must be identical cluster-wide for all identical replicants
    * @param subscriber The unsubscribing {@link ReplicantListener}
    */   
   public void unregisterListener(String key, ReplicantListener subscriber);

   // State binding methods
   //

   /**
    * Add a replicant, which will be associated with this cluster node in the registry.
    * @param key Replicant name. All related replicas around the cluster must use the same key name.
    * @param replicant Local data of the replicant, that is, any serializable data
    * @throws Exception Thrown if a cluster communication problem occurs
    */
   public void add(String key, Serializable replicant) throws Exception;

   /**
    * Remove this cluster node's replicant for the given key from the registry
    * @param key Name of the replicant
    * @throws Exception Thrown if a cluster communication problem occurs
    */
   public void remove(String key) throws Exception;

   /**
    * Lookup the replicant associated with this cluster node and the given key
    * @param key The name of the replicant
    * @return The local replicant for the given key name
    */   
   public Serializable lookupLocalReplicant(String key);

   /**
    * Return a list of all replicants for the given key.
    * @param key The replicant name
    * @return A list of serialized replicants available around the cluster 
    *         for the given key. This list will be in the same order in all 
    *         nodes in the cluster.
    */
   public List lookupReplicants(String key);

   /**
    * Return a list of the names of all nodes that have registered a replicant
    * for the given key.
    * @param key The replicant name
    * @return A list of the node names of cluster nodes that have made available
    *         a replicant for the given key. This list will be in the same 
    *         order in all nodes in the cluster.
    * @deprecated Replaced by {@link #lookupReplicantsNodes(String)} that 
    *             returns a List of ClusterNode. Call 
    *             {@link ClusterNode#getName()} on List entries to get same 
    *             behavior. 
    */
   @Deprecated public List lookupReplicantsNodeNames(String key);
   
   /**
    * Return a list of the {@link ClusterNode} objects for  all nodes that have 
    * registered a replicant for the given key.
    * @param key The replicant name
    * @return A list of the {@link ClusterNode} instances that have made available
    *         a replicant for the given key. This list will be in the same 
    *         order in all nodes in the cluster.
    */
   public List<ClusterNode> lookupReplicantsNodes(String key);

   /**
    * Return a list of all services that have at least one replicant.
    * @return A <code>Collection<String></code> of service names; i.e. the 
    *         <code>key</code>s passed to {@link #add(String, Serializable)}
    */
   public Collection getAllServices ();

   /**
    * Returns an id corresponding to the current view of this set of replicants.
    * The id is a hash of the list of nodes that have registered a replicant 
    * for the key.
    * 
    * @param key The replicant name
    * @return A view id (doesn't grow sequentially)
    */
   public int getReplicantsViewId(String key);
   
   /**
    * Indicates if the current node is the master replica for this given key; 
    * i.e. whether the current node would be the first element in the list 
    * returned from a call to <code>lookupReplicantsNodes(key)</code>.
    * 
    * @param key The replicant name
    * @return True if this node is the master
    */
   public boolean isMasterReplica (String key);

}
