/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.ha.framework.server.lock;

import java.io.Serializable;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.LockSupport;

import org.jboss.ha.framework.interfaces.ClusterNode;
import org.jboss.ha.framework.interfaces.HAPartition;

/**
 * Distributed lock manager intended for use cases where multiple local threads
 * can share the lock, but only one node in the cluster can have threads using
 * the lock. Nodes holding the lock yield it to remote requestors if no local
 * threads are using it; otherwise remote requestors block.
 * <p>
 * The expected use case for this class is controlling access to resources that
 * are typically only accessed on a single node (e.g. web sessions or stateful
 * session beans), with the distributed lock used primarily to guarantee that.
 * </p>
 * @author Brian Stansberry
 * 
 * @version $Revision:$
 */
public class SharedLocalYieldingClusterLockManager
{
   /** Result of a {@link SharedLocalYieldingClusterLockManager#lock(Serializable, long, boolean) lock call}*/
   public static enum LockResult 
   { 
      /** Indicates the lock was acquired after requesting it from the cluster */
      ACQUIRED_FROM_CLUSTER, 
      /** Indicates this node already held the lock */
      ALREADY_HELD, 
      /** 
       * Indicates the 'newLock' param passed to 
       * {@link YieldingClusterLockManager#lock(Serializable, long, boolean)}
       * was <code>true</code> and the local node in fact was unaware of the
       * lock. If in fact the local node was already aware of the lock (which
       * would generally indicate a flaw in the application using this class)
       * NEW_LOCK will not be returned; rather one of the other enum values
       * will be returned.
       */
      NEW_LOCK 
   }
   
   private class LocalLock
   {
      private volatile ClusterNode holder;
      private volatile boolean removable;
      private final Queue<Thread> waiters = new ConcurrentLinkedQueue<Thread>();
      private final AtomicInteger localLockCount = new AtomicInteger();

      private void lock(ClusterNode caller, long timeout) throws TimeoutException
      {
         if (SharedLocalYieldingClusterLockManager.this.localNode.equals(caller))
         {
            this.localLockCount.incrementAndGet();
            this.holder = SharedLocalYieldingClusterLockManager.this.localNode;
         }
         else
         {
            long deadline = System.currentTimeMillis() + timeout;
            boolean wasInterrupted = false;
            Thread current = Thread.currentThread();
            waiters.add(current);
            
            try
            {
               // Block while not first in queue or cannot acquire lock
               while (waiters.peek() != current || 
                      localLockCount.get() > 0) 
               { 
                  LockSupport.parkUntil(deadline);
                  if (Thread.interrupted()) // ignore interrupts while waiting
                     wasInterrupted = true;
                  if (System.currentTimeMillis() >= deadline)
                  {
                     if (waiters.peek() != current || 
                           localLockCount.get() > 0)
                     {
                        throw new TimeoutException(this.holder);
                     }
                     break;
                  }
               }
               
               if (localLockCount.get() == 0)
               {
                  holder = caller;
               }
               else
               {
                  throw new TimeoutException(this.holder);
               }
            }
            finally
            {
               waiters.remove();
               if (wasInterrupted)          // reassert interrupt status on exit
                  current.interrupt();
            }
         }
      }
      
      private void unlock(ClusterNode caller)
      {
         if (caller.equals(holder))              
         {
            if (SharedLocalYieldingClusterLockManager.this.localNode.equals(caller))
            {
               if (this.localLockCount.decrementAndGet() == 0)
               {
                  holder = null;
               }
            }
            else
            {
               holder = null;
            }
            
            if (holder == null)
            {
               LockSupport.unpark(waiters.peek());
            }
         }
       } 
      
   }
   
   /** Handles callbacks from the cluster lock support object */
   private class ClusterHandler implements LocalLockHandler
   {      
      // ----------------------------------------------------- LocalLockHandler
      
      public ClusterNode getLocalNode(ClusterNode localNode)
      {
         return SharedLocalYieldingClusterLockManager.this.localNode;
      }

      public void setLocalNode(ClusterNode localNode)
      {
         SharedLocalYieldingClusterLockManager.this.localNode = localNode;
      }

      public void lockFromCluster(Serializable lockName, ClusterNode caller, long timeout) throws TimeoutException,
            InterruptedException
      {
         LocalLock lock = getLocalLock(lockName, true);
         lock.lock(caller, timeout);
         if (!localNode.equals(caller))
         {
            // Any local thread who has a ref to lock will now need to request it
            // remotely from caller, which won't grant it until this method returns.
            // So, we can remove lock from the map. If that local thread is granted
            // the lock by caller, when that thread calls lockFromCluster, we'll create
            // a new lock to handle that.
            localLocks.remove(lockName, lock);
         }
      }

      public ClusterNode getLockHolder(Serializable lockName)
      {
         LocalLock lock = getLocalLock(lockName, false);
         return lock == null ? null : lock.holder;
      }

      public void unlockFromCluster(Serializable lockName, ClusterNode caller)
      {
         LocalLock lock = getLocalLock(lockName, false);
         if (lock != null)
         {
            lock.unlock(caller);
         }
      }
      
   }
   
   private ClusterNode localNode;
   private ConcurrentMap<Serializable, LocalLock> localLocks = new ConcurrentHashMap<Serializable, LocalLock>();
   private final YieldingGloballyExclusiveClusterLockSupport clusterSupport;
   
   public SharedLocalYieldingClusterLockManager(String serviceHAName, HAPartition partition)
   {
      ClusterHandler handler = new ClusterHandler();
      clusterSupport = new YieldingGloballyExclusiveClusterLockSupport(serviceHAName, partition, handler);
   }
   
   // ----------------------------------------------------------------- Public
   
   /**
    * Acquire the given lock.
    * 
    * @param lockName the identifier of the lock that should be acquired
    * @param timeout max time in ms to wait before throwing a TimeoutException
    *                if the lock cannot be acquired
    *                
    * @return enum indicating how the lock was acquired
    *                
    * @throws TimeoutException if the lock cannot be acquired before the timeout
    * @throws InterruptedException if the thread is interrupted while trying to
    *                              acquire the lock
    */
   public LockResult lock (Serializable lockName, long timeout)
      throws TimeoutException, InterruptedException
   {
      return lock(lockName, timeout, false);
   }
   
   /**
    * Acquire the given lock.
    * 
    * @param lockName the identifier of the lock that should be acquired
    * @param timeout max time in ms to wait before throwing a TimeoutException
    *                if the lock cannot be acquired
    * @param newLock <code>true</code> if this object should assume this is the
    *                first use cluster-wide of the lock identified by
    *                <code>lockName</code>, and just acquire the lock locally 
    *                without any cluster-wide call. See discussion of 
    *                {@link LockResult#NEW_LOCK}.
    *                
    * @return enum indicating how the lock was acquired
    *                
    * @throws TimeoutException if the lock cannot be acquired before the timeout
    * @throws InterruptedException if the thread is interrupted while trying to
    *                              acquire the lock
    */
   public LockResult lock(Serializable lockName, long timeout, boolean newLock)
      throws TimeoutException, InterruptedException
   {   
      if (this.localNode == null)
      {
         throw new IllegalStateException("Null localNode");
      }
      
      LockResult result = null;
      LocalLock localLock = getLocalLock(lockName, false);
      if (localLock == null)
      {
         if (newLock)
         {
            // Here we assume the caller knows what they are doing and this 
            // is really is a new lock, and that no other
            // node is going to try to take it
            localLock = getLocalLock(lockName, true);
            localLock.lock(this.localNode, timeout);
            result = (localLock.localLockCount.get() == 1 ? LockResult.NEW_LOCK : LockResult.ALREADY_HELD);
         }
         else
         {
            this.clusterSupport.lock(lockName, timeout);
            result = LockResult.ACQUIRED_FROM_CLUSTER;
         }
      }
      else
      {
         localLock.localLockCount.incrementAndGet(); // Now no other node can become localLock.holder         
         try
         {
            if (this.localNode.equals(localLock.holder))
            {
               result = LockResult.ALREADY_HELD;
               
               // Check for race where we locked something that's been removed
               if (localLock.removable && localLock != getLocalLock(lockName, false))
               {
                  return lock(lockName, timeout, newLock);
               }
            }
            else
            {
               this.clusterSupport.lock(lockName, timeout);
               result = LockResult.ACQUIRED_FROM_CLUSTER;
            }
         }
         finally
         {
            // If we called clusterSupport.lock() above, its callback into
            // ClusterHandler.lockFromCluster() will increment localLock.localLockCount.
            // So, decrement so we don't double count
            // (If we threw an exception above we should also decrement)
            if (result != LockResult.ALREADY_HELD)
            {
               // Only decrement if the current lock object for this key is
               // the same one we incremented above
               LocalLock current = localLocks.get(lockName);
               if (current == localLock)
               {
                  localLock.localLockCount.decrementAndGet();
               }
            }
         }         
      }
      
      return result;
   }
   
   /**
    * Releases a previously acquired lock.
    * 
    * @param lockName unique name identifying the lock to release
    * @param remove <code>true</code> if this lock can be removed from
    *               tracking once all local locks are unlocked.
    */
   public void unlock(Serializable lockName, boolean remove)
   {
      LocalLock lock = getLocalLock(lockName, false);
      if (remove && lock != null)
      {
         lock.removable = true;
      }
      
      this.clusterSupport.unlock(lockName);
      
      if (lock != null && lock.removable && lock.localLockCount.get() == 0)
      {
         localLocks.remove(lockName, lock);
      }
   }
   
   /**
    * Brings this object to a state where it is ready for normal operation.
    * 
    * @throws Exception
    */
   public void start() throws Exception
   {
      this.clusterSupport.start();
   }
   
   /**
    * Removes this object from a state where it is ready for normal oepration
    * and performs cleanup work.
    * 
    * @throws Exception
    */
   public void stop() throws Exception
   {
      this.clusterSupport.stop();
   }
   
   // ----------------------------------------------------------------- Private
   
   private LocalLock getLocalLock(Serializable categoryName, boolean create)
   {
      LocalLock category = localLocks.get(categoryName);
      if (category == null && create)
      {
         category = new LocalLock();
         LocalLock existing = localLocks.putIfAbsent(categoryName, category);
         if (existing != null)
         {
            category = existing;
         }         
      }
      return category;
   }

}
