/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.tomcat.service.session.distributedcache.ispn;

import org.infinispan.Cache;
import org.infinispan.config.Configuration;
import org.infinispan.config.Configuration.CacheMode;
import org.infinispan.manager.EmbeddedCacheManager;
import org.jboss.ha.ispn.CacheContainerRegistry;
import org.jboss.metadata.web.jboss.ReplicationConfig;
import org.jboss.metadata.web.jboss.ReplicationMode;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;

/**
 * @author Paul Ferraro
 *
 */
public class DefaultCacheSource implements CacheSource
{
   private static final String DELIMITER = "/";
   
   private final CacheContainerRegistry registry;
   
   public DefaultCacheSource(CacheContainerRegistry registry)
   {
      this.registry = registry;
   }
   
   @Override
   public <K, V> Cache<K, V> getCache(LocalDistributableSessionManager manager)
   {
      ReplicationConfig config = manager.getReplicationConfig();
      String containerName = config.getCacheName();
      String templateCacheName = null;
      
      if ((containerName != null) && !containerName.isEmpty())
      {
         String[] parts = containerName.split(DELIMITER);
         
         if (parts.length == 2)
         {
            containerName = parts[0];
            templateCacheName = parts[1];
         }
      }
      
      String cacheName = manager.getName();
      
      EmbeddedCacheManager container = this.registry.getCacheContainer(containerName);
      
      this.applyOverrides(container.defineConfiguration(cacheName, templateCacheName, new Configuration()), config);
      
      return container.<K, V>getCache(cacheName);
   }

   private Configuration applyOverrides(Configuration configuration, ReplicationConfig replConfig)
   {
      Integer backups = replConfig.getBackups();
      ReplicationMode replMode = replConfig.getReplicationMode();
      
      CacheMode mode = configuration.getCacheMode();
      
      if (backups != null)
      {
         int value = backups.intValue();
         
         configuration.setNumOwners(value);
         
         if (value == 0)
         {
            mode = CacheMode.LOCAL;
         }
         else
         {
            boolean dist = (value > 0);
            boolean synchronous = mode.isSynchronous();
            if (dist)
            {
               mode = synchronous ? CacheMode.DIST_SYNC : CacheMode.DIST_ASYNC;
            }
            else // Negative backups means total replication
            {
               mode = synchronous ? CacheMode.REPL_SYNC : CacheMode.REPL_ASYNC;
            }
            configuration.setFetchInMemoryState(!dist);
         }
      }
      
      if (replMode != null)
      {
         switch (replMode)
         {
            case SYNCHRONOUS:
            {
               mode = mode.toSync();
               break;
            }
            case ASYNCHRONOUS:
            {
               mode = mode.toAsync();
               break;
            }
         }
      }
      
      configuration.setCacheMode(mode);
      
      return configuration;
   }
}
