/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.tomcat.service.session.distributedcache.ispn;

import javax.transaction.TransactionManager;

import org.infinispan.Cache;
import org.infinispan.atomic.AtomicMap;
import org.infinispan.transaction.tm.BatchModeTransactionManager;
import org.jboss.ha.framework.server.lock.SharedLocalYieldingClusterLockManager;
import org.jboss.ha.ispn.DefaultCacheContainerRegistry;
import org.jboss.ha.ispn.atomic.AtomicMapFactory;
import org.jboss.ha.ispn.atomic.DefaultAtomicMapFactory;
import org.jboss.ha.ispn.invoker.CacheInvoker;
import org.jboss.ha.ispn.invoker.RetryingCacheInvoker;
import org.jboss.web.tomcat.service.session.distributedcache.impl.BatchingManagerImpl;
import org.jboss.web.tomcat.service.session.distributedcache.impl.SessionAttributeMarshallerFactoryImpl;
import org.jboss.web.tomcat.service.session.distributedcache.spi.BatchingManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.OutgoingDistributableSessionData;
import org.jboss.web.tomcat.service.session.distributedcache.spi.SessionAttributeMarshaller;
import org.jboss.web.tomcat.service.session.distributedcache.spi.SessionAttributeMarshallerFactory;

/**
 * Factory for creating an Infinispan-backed distributed cache manager.
 * @author Paul Ferraro
 */
public class DistributedCacheManagerFactory implements org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManagerFactory
{
   private CacheSource cacheSource = new DefaultCacheSource(DefaultCacheContainerRegistry.getInstance());
   private LockManagerSource lockManagerSource = new DefaultLockManagerSource();
   private SessionAttributeStorageFactory storageFactory = new SessionAttributeStorageFactoryImpl();
   private SessionAttributeMarshallerFactory marshallerFactory = new SessionAttributeMarshallerFactoryImpl();
   private CacheInvoker invoker = new RetryingCacheInvoker(10, 100);
   private AtomicMapFactory atomicMapFactory = new DefaultAtomicMapFactory();

   @Override
   public <T extends OutgoingDistributableSessionData> org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager<T> getDistributedCacheManager(LocalDistributableSessionManager manager)
   {
      Cache<String, AtomicMap<Object, Object>> cache = this.cacheSource.getCache(manager);
      SharedLocalYieldingClusterLockManager lockManager = this.lockManagerSource.getLockManager(cache);
      TransactionManager tm = cache.getAdvancedCache().getTransactionManager();
      
      if (!(tm instanceof BatchModeTransactionManager))
      {
         throw new IllegalStateException("Unexpected transaction manager type: " + ((tm != null) ? tm.getClass().getName() : "null"));
      }
      
      BatchingManager batchingManager = new BatchingManagerImpl(tm);
      SessionAttributeMarshaller marshaller = this.marshallerFactory.createMarshaller(manager);
      SessionAttributeStorage<T> storage = this.storageFactory.createStorage(manager.getReplicationConfig().getReplicationGranularity(), marshaller);
      
      return new DistributedCacheManager<T>(manager, cache, lockManager, storage, batchingManager, this.invoker, this.atomicMapFactory);
   }
   
   public void setCacheSource(CacheSource source)
   {
      this.cacheSource = source;
   }
   
   public void setLockManagerSource(LockManagerSource source)
   {
      this.lockManagerSource = source;
   }
   
   public void setSessionAttributeStorageFactory(SessionAttributeStorageFactory storageFactory)
   {
      this.storageFactory = storageFactory;
   }
   
   public void setSessionAttributeMarshallerFactory(SessionAttributeMarshallerFactory marshallerFactory)
   {
      this.marshallerFactory = marshallerFactory;
   }
   
   public void setCacheInvoker(CacheInvoker invoker)
   {
      this.invoker = invoker;
   }
   
   public void setAtomicMapFactory(AtomicMapFactory atomicMapFactory)
   {
      this.atomicMapFactory = atomicMapFactory;
   }
}
