/*
 * Copyright 2016-2019 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.galleon.xml.test;

import java.nio.file.Paths;
import java.util.Locale;

import org.jboss.galleon.universe.FeaturePackLocation;
import org.jboss.galleon.config.FeaturePackConfig;
import org.jboss.galleon.config.ProvisioningConfig;
import org.jboss.galleon.test.util.XmlParserValidator;
import org.jboss.galleon.xml.ProvisioningXmlParser;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * @author <a href="https://github.com/ppalaga">Peter Palaga</a>
 */
public class ProvisioningXmlParserTestCase {

    private static final XmlParserValidator<ProvisioningConfig> validator = new XmlParserValidator<>(
            Paths.get("src/main/resources/schema/galleon-provisioning-3_0.xsd"), ProvisioningXmlParser.getInstance());

    private static final Locale defaultLocale = Locale.getDefault();

    @BeforeClass
    public static void setLocale() {
        Locale.setDefault(Locale.US);
    }
    @AfterClass
    public static void resetLocale() {
        Locale.setDefault(defaultLocale);
    }

    @Test
    public void readBadNamespace() throws Exception {
        /*
         * urn:jboss:galleon:provisioning:1.0.1 used in provisioning-1.0.1.xml is not registered in ProvisioningXmlParser
         */
        validator.validateAndParse("xml/provisioning/provisioning-1.0.1.xml",
                "Cannot find the declaration of element 'installation'.",
                "Message: Unexpected element '{urn:jboss:galleon:provisioning:1.0.1}installation'");
    }

    @Test
    public void readNoFp() throws Exception {
        validator.validateAndParse("xml/provisioning/provisioning-1.0-no-fp.xml", null, null);
    }

    @Test
    public void readValid() throws Exception {
        ProvisioningConfig found = validator
                .validateAndParse("xml/provisioning/provisioning.xml", null, null);
        ProvisioningConfig expected = ProvisioningConfig.builder()
                .addUniverse("universe1", "factory1", "location1")
                .addUniverse(null, "factory2", "location2")
                .addUniverse("universe3", "factory3", "location3")
                .addFeaturePackDep(FeaturePackConfig.forTransitiveDep(FeaturePackLocation.fromString("fp4@maven(universe):0#0.0.4")))
                .addFeaturePackDep(FeaturePackConfig.forTransitiveDep(FeaturePackLocation.fromString("fp5@maven(universe):0#0.0.5")))
                .addFeaturePackDep(FeaturePackLocation.fromString("fp1@maven(universe):0#0.0.1"))
                .addFeaturePackDep(FeaturePackLocation.fromString("fp2@maven(universe):0#0.0.2"))
                .addFeaturePackDep(FeaturePackLocation.fromString("fp3@maven(universe):0#0.0.3"))
                .setInheritConfigs(false)
                .setInheritModelOnlyConfigs(false)
                .includeDefaultConfig("model1", "name1")
                .includeDefaultConfig(null, "name2")
                .excludeDefaultConfig("model1", "name2")
                .excludeDefaultConfig(null, "name1")
                .addOption("name1", "value1")
                .addOption("name2", "value2")
                .build();
        Assert.assertEquals(expected, found);
    }

}
