/*
 * Copyright 2015-2016 Red Hat, Inc, and individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.hal.client.deployment;

import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import elemental.js.util.JsArrayOf;
import org.jboss.hal.ballroom.tree.Node;
import org.jboss.hal.dmr.ModelNode;
import org.jboss.hal.resources.Ids;

import static org.jboss.hal.dmr.ModelDescriptionConstants.PATH;
import static org.jboss.hal.resources.CSS.fontAwesome;

/**
 * @author Harald Pehl
 */
class ContentParser {

    private static final Comparator<ContentEntry> BY_NAME = (c1, c2) -> c1.name.compareTo(c2.name);
    private static final Comparator<ContentEntry> BY_DEPTH = (c1, c2) -> Integer.compare(c1.depth, c2.depth);

    private static final String DIRECTORY = "directory";
    private static final String FILE_SIZE = "file-size";

    void parse(JsArrayOf<Node<ContentEntry>> nodes, Node<ContentEntry> root, List<ModelNode> content) {
        nodes.push(root);

        Map<String, Node<ContentEntry>> nodesByPath = new HashMap<>();
        content.stream()
                .map(this::contentEntry)
                .filter(contentEntry -> contentEntry.directory)
                .sorted(BY_DEPTH.thenComparing(BY_NAME))
                .forEach(directory -> {
                    String parentPath = parentPath(directory);
                    Node<ContentEntry> parent = parentPath == null ? root : nodesByPath.get(parentPath);

                    if (parent != null) {
                        Node<ContentEntry> node = pushFolder(nodes, parent, directory);
                        nodesByPath.put(directory.path, node);
                    }
                });

        content.stream()
                .map(this::contentEntry)
                .filter(contentEntry -> !contentEntry.directory)
                .sorted(BY_NAME)
                .forEach(file -> {
                    String parentPath = parentPath(file);
                    Node<ContentEntry> parent = parentPath == null ? root : nodesByPath.get(parentPath);
                    if (parent != null) {
                        pushEntry(nodes, parent, file);
                    }
                });
    }

    private ContentEntry contentEntry(ModelNode node) {
        String path = node.get(PATH).asString();
        Iterable<String> segments = Splitter.on('/').omitEmptyStrings().split(path);

        ContentEntry contentEntry = new ContentEntry();
        contentEntry.name = Iterables.getLast(segments);
        contentEntry.path = path;
        contentEntry.depth = Iterables.size(segments);
        contentEntry.directory = node.hasDefined(DIRECTORY) && node.get(DIRECTORY).asBoolean();
        contentEntry.fileSize = node.hasDefined(FILE_SIZE) ? node.get(FILE_SIZE).asLong() : 0;
        return contentEntry;
    }

    private Node<ContentEntry> pushFolder(JsArrayOf<Node<ContentEntry>> nodes, Node<ContentEntry> parent,
            ContentEntry contentEntry) {
        Node<ContentEntry> node = new Node.Builder<>(Ids.build(parent.id, Ids.uniqueId()), contentEntry.name,
                contentEntry)
                .parent(parent.id)
                .folder()
                .build();
        nodes.push(node);
        return node;
    }

    private Node<ContentEntry> pushEntry(JsArrayOf<Node<ContentEntry>> nodes, Node<ContentEntry> parent,
            ContentEntry contentEntry) {
        Node<ContentEntry> node = new Node.Builder<>(Ids.build(parent.id, Ids.uniqueId()), contentEntry.name,
                contentEntry)
                .parent(parent.id)
                .icon(fontAwesome("file-text-o"))
                .build();
        nodes.push(node);
        return node;
    }

    private String parentPath(ContentEntry contentEntry) {
        String path = contentEntry.path.endsWith("/")
                ? contentEntry.path.substring(0, contentEntry.path.length() - 1)
                : contentEntry.path;
        int index = path.lastIndexOf('/');
        if (index != -1) {
            return path.substring(0, index + 1);
        }
        return null;
    }
}
