/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api.session.managers;

import org.jboss.identity.idm.api.PersistenceManager;
import org.jboss.identity.idm.api.User;
import org.jboss.identity.idm.api.Group;
import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.api.PersistenceManagerFeaturesDescription;
import org.jboss.identity.idm.api.IdentitySearchCriteria;
import org.jboss.identity.idm.api.IdentitySearchCriteriumType;
import org.jboss.identity.idm.common.exception.IdentityException;
import org.jboss.identity.idm.spi.model.IdentityObjectType;
import org.jboss.identity.idm.spi.model.IdentityObject;
import org.jboss.identity.idm.spi.store.IdentityObjectSearchCriteriaType;
import org.jboss.identity.idm.impl.api.session.managers.AbstractManager;
import org.jboss.identity.idm.impl.api.model.GroupId;

import java.util.Collection;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.io.Serializable;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class PersistenceManagerImpl extends AbstractManager implements PersistenceManager, Serializable
{

   private final PersistenceManagerFeaturesDescription featuresDescription;
   
   private static final long serialVersionUID = -4691446225503953920L;


   public PersistenceManagerImpl(IdentitySession session)
   {
      super(session);

      featuresDescription = new PersistenceManagerFeaturesDescription()
      {
         public boolean isUsersAddRemoveSupported()
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
               isIdentityObjectAddRemoveSupported(objectType);
         }

         public boolean isGroupsAddRemoveSupported(String groupType)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(groupType);

            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
               isIdentityObjectAddRemoveSupported(objectType);
         }

         public boolean isUsersSearchCriteriumTypeSupported(IdentitySearchCriteriumType constraintType)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

            IdentityObjectSearchCriteriaType constraint = IdentityObjectSearchCriteriaType.valueOf(constraintType.name());


            if (constraint != null)
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isSearchCriteriaTypeSupported(objectType, constraint);
            }

            return false;
         }


         public boolean isGroupsSearchCriteriumTypeSupported(String groupType, IdentitySearchCriteriumType constraintType)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(groupType);

            IdentityObjectSearchCriteriaType constraint = IdentityObjectSearchCriteriaType.valueOf(constraintType.name());

            if (constraint != null)
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isSearchCriteriaTypeSupported(objectType, constraint);
            }

            return false;
         }

      };
   }

   public PersistenceManagerFeaturesDescription getFeaturesDescription()
   {
      return featuresDescription;
   }

   public User createUser(String identityName) throws IdentityException
   {
      checkNotNullArgument(identityName, "Identity name");
      checkObjectName(identityName);

      IdentityObjectType iot = getUserObjectType();

      IdentityObject identityObject = getRepository().createIdentityObject(getInvocationContext(), identityName, iot);

      return createUser(identityObject);
   }

   public Group createGroup(String groupName, String groupType) throws IdentityException
   {
      checkNotNullArgument(groupName, "Group name");
      checkNotNullArgument(groupType, "Group type");
      checkObjectName(groupName);
      checkObjectName(groupType);

      IdentityObjectType iot = getIdentityObjectType(groupType);

      IdentityObject identityObject = getRepository().createIdentityObject(getInvocationContext(), groupName, iot);

      return createGroup(identityObject);
   }

   public String createGroupId(String groupName, String groupType)
   {
      checkNotNullArgument(groupName, "Group name");
      checkNotNullArgument(groupType, "Group type");
      checkObjectName(groupName);
      checkObjectName(groupType);

      return new GroupId(groupName, groupType).getId();
   }

   public void removeUser(User user, boolean force) throws IdentityException
   {
      checkNotNullArgument(user, "User");
      getRepository().removeIdentityObject(getInvocationContext(), createIdentityObject(user));
   }

   public void removeUser(String userName, boolean force) throws IdentityException
   {
      checkNotNullArgument(userName, "User name");
      getRepository().removeIdentityObject(getInvocationContext(), createIdentityObjectForUserName(userName));
   }

   public void removeGroup(Group group, boolean force) throws IdentityException
   {
      checkNotNullArgument(group, "Group");

      //TODO: force

      getRepository().removeIdentityObject(getInvocationContext(), createIdentityObject(group));
   }

   public void removeGroup(String groupId, boolean force) throws IdentityException
   {
      checkNotNullArgument(groupId, "Group Id");

      //TODO: force

      getRepository().removeIdentityObject(getInvocationContext(), createIdentityObjectForGroupId(groupId));
   }

   public int getUserCount() throws IdentityException
   {
      IdentityObjectType iot = getUserObjectType();

      return getRepository().getIdentityObjectsCount(getInvocationContext(), iot);
   }

   public int getGroupTypeCount(String groupType) throws IdentityException
   {
      checkNotNullArgument(groupType, "Group type");

      IdentityObjectType iot = getIdentityObjectType(groupType);

      return getRepository().getIdentityObjectsCount(getInvocationContext(), iot);
   }

   public User findUser(String name) throws IdentityException
   {
      checkNotNullArgument(name, "User name");

      IdentityObject io = getRepository().findIdentityObject(getInvocationContext(), name, getUserObjectType());

      if (io != null)
      {
         return createUser(io);
      }
      return null;
   }

   public Collection<User> findUser(IdentitySearchCriteria criteria) throws IdentityException
   {


      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), getUserObjectType(), convertSearchControls(criteria));
      List<User> identities = new LinkedList<User>();

      for (Iterator<IdentityObject> iterator = ios.iterator(); iterator.hasNext();)
      {
         IdentityObject identityObject = iterator.next();
         identities.add(createUser(identityObject));
      }

      return identities;
   }

   public Group findGroup(String name, String groupType) throws IdentityException
   {
      checkNotNullArgument(name, "Group name");
      checkNotNullArgument(groupType, "Group type");

      IdentityObject io = getRepository().findIdentityObject(getInvocationContext(), name, getIdentityObjectType(groupType));

      if (io != null)
      {
         return createGroup(io);
      }
      return null; 
   }

   public Group findGroupById(String id) throws IdentityException
   {
      checkNotNullArgument(id, "Group id");

      IdentityObject io =  getRepository().findIdentityObject(getInvocationContext(), id);

      if (io != null)
      {
         return createGroup(io);
      }
      return null;
   }

   public Collection<Group> findGroup(String groupType, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(groupType, "Group type");

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), getIdentityObjectType(groupType), convertSearchControls(criteria));
      List<Group> groups = new LinkedList<Group>();

      for (Iterator<IdentityObject> iterator = ios.iterator(); iterator.hasNext();)
      {
         IdentityObject identityObject = iterator.next();
         groups.add(createGroup(identityObject));
      }

      return groups;
   }

   public Collection<Group> findGroup(String groupType) throws IdentityException
   {
      checkNotNullArgument(groupType, "Group type");

      return findGroup(groupType, (IdentitySearchCriteria) null);
   }

//   public boolean isVirtual(User user)
//   {
//      //TODO:NYI
//      throw new NotYetImplementedException("Postponed");
//   }
}
