/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api.session.managers;

import org.jboss.identity.idm.api.RoleManager;
import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.api.RoleType;
import org.jboss.identity.idm.api.Role;
import org.jboss.identity.idm.api.User;
import org.jboss.identity.idm.api.Group;
import org.jboss.identity.idm.api.IdentityType;
import org.jboss.identity.idm.api.RoleManagerFeaturesDescription;
import org.jboss.identity.idm.api.IdentitySearchCriteria;
import org.jboss.identity.idm.api.IdentitySearchCriteriumType;
import org.jboss.identity.idm.common.exception.IdentityException;
import org.jboss.identity.idm.spi.model.IdentityObjectRelationshipType;
import org.jboss.identity.idm.spi.model.IdentityObjectRelationship;
import org.jboss.identity.idm.spi.model.IdentityObjectType;
import org.jboss.identity.idm.spi.model.IdentityObject;
import org.jboss.identity.idm.spi.exception.OperationNotSupportedException;
import org.jboss.identity.idm.spi.store.IdentityObjectSearchCriteriaType;
import org.jboss.identity.idm.impl.api.model.SimpleRoleType;
import org.jboss.identity.idm.impl.api.model.SimpleRole;
import org.jboss.identity.idm.impl.types.SimpleIdentityObjectRelationship;

import java.util.Collection;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.HashMap;
import java.io.Serializable;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class RoleManagerImpl extends AbstractManager implements RoleManager, Serializable
{
   public static final IdentityObjectRelationshipType ROLE = new IdentityObjectRelationshipType()
   {
      public String getName()
      {
         return "JBOSS_IDENTITY_ROLE";
      }
   };

   private final RoleManagerFeaturesDescription featuresDescription;
   
   private static final long serialVersionUID = 7246982831145808636L;

   public RoleManagerImpl(IdentitySession session)
   {
      super(session);

      featuresDescription = new RoleManagerFeaturesDescription()
      {
         public boolean isRoleTypeAddRemoveSupported()
         {
            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().isRelationshipNameAddRemoveSupported();
         }

         public boolean isRoleTypeSearchCriteriumTypeSupported(IdentitySearchCriteriumType constraintType)
         {
            IdentityObjectSearchCriteriaType constraint = IdentityObjectSearchCriteriaType.valueOf(constraintType.name());

            if (constraint != null)
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isRoleNameSearchCriteriaTypeSupported(constraint);
            }
            else
            {
               return false;
            }
         }
      };
   }

   protected IdentityObjectRelationship createIdentityObjectRelationship(Role role)
   {
      return new SimpleIdentityObjectRelationship(
         createIdentityObject(role.getGroup()),
         createIdentityObject(role.getUser()),
         role.getRoleType().getName(),
         ROLE
      );
   }

   public RoleManagerFeaturesDescription getSupportedFeatures()
   {
      return featuresDescription;
   }

   public RoleType createRoleType(String name) throws IdentityException
   {
      checkNotNullArgument(name, "RoleType name");
      checkObjectName(name);

      String roleType = null;

      try
      {
         roleType = getRepository().createRelationshipName(getInvocationContext(), name);
      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported");
      }

      return new SimpleRoleType(roleType);
   }

   public void removeRoleType(String name) throws IdentityException
   {
      checkNotNullArgument(name, "RoleType name");

      try
      {
         getRepository().removeRelationshipName(getInvocationContext(), name);
      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported");
      }
   }

   public void removeRoleType(RoleType roleType) throws IdentityException
   {
      checkNotNullArgument(roleType, "RoleType");

      removeRoleType(roleType.getName());
   }

   public RoleType getRoleType(String name) throws IdentityException
   {

      checkNotNullArgument(name, "RoleType name");
      try
      {
         Set<String> names = getRepository().getRelationshipNames(getInvocationContext(), null);
         if (names.contains(name))
         {
            return new SimpleRoleType(name);
         }
      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported");
      }

      return null;
   }

   public Collection<RoleType> findRoleTypes(IdentitySearchCriteria criteria) throws IdentityException
   {

      
      try
      {
         Set<String> names = getRepository().getRelationshipNames(getInvocationContext(), convertSearchControls(criteria));
         Set<RoleType> types = new HashSet<RoleType>();

         for (String name : names)
         {
            types.add(new SimpleRoleType(name));
         }

         return types;
      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported");
      }

   }

   public Collection<RoleType> findRoleTypes() throws IdentityException
   {
      return findRoleTypes(null);
   }

   public Role createRole(RoleType roleType, User user, Group group) throws IdentityException
   {
      checkNotNullArgument(roleType, "RoleType");
      checkNotNullArgument(user, "User");
      checkNotNullArgument(group, "Group");

      //TODO: add createRoleType switch to the API

      IdentityObjectRelationship rel = getRepository().createRelationship(getInvocationContext(), createIdentityObject(group), createIdentityObject(user), ROLE, roleType.getName(), false);

      //TODO: null id - IdentityObjectRelationship doesn't have id
      return new SimpleRole(new SimpleRoleType(rel.getName()), createUser(rel.getToIdentityObject()), createGroup(rel.getFromIdentityObject()));

   }

   public Role createRole(String roleTypeName, String userName, String groupId) throws IdentityException
   {
      checkNotNullArgument(roleTypeName, "RoleType name");
      checkNotNullArgument(userName, "User name");
      checkNotNullArgument(groupId, "Group Id");

      User user = createUserFromId(userName);
      Group group = createGroupFromId(groupId);

      return createRole(new SimpleRoleType(roleTypeName), user, group);
   }

   public void removeRole(RoleType roleType, User user, Group group) throws IdentityException
   {
      checkNotNullArgument(roleType, "RoleType");
      checkNotNullArgument(user, "User");
      checkNotNullArgument(group, "Group");

      getRepository().removeRelationship(getInvocationContext(), createIdentityObject(group), createIdentityObject(user), ROLE, roleType.getName());
   }

   public void removeRole(String roleTypeName, String userName, String groupId) throws IdentityException
   {
      checkNotNullArgument(roleTypeName, "RoleType name");
      checkNotNullArgument(userName, "User name");
      checkNotNullArgument(groupId, "Group Id");

      User user = createUserFromId(userName);
      Group group = createGroupFromId(groupId);

      removeRole(new SimpleRoleType(roleTypeName), user, group);
   }

   public void removeRole(Role role) throws IdentityException
   {
      checkNotNullArgument(role, "Role");

      getRepository().removeRelationship(getInvocationContext(), createIdentityObject(role.getGroup()), createIdentityObject(role.getUser()), ROLE, role.getRoleType().getName());
   }

   public boolean hasRole(User user, Group group, RoleType roleType) throws IdentityException
   {
      checkNotNullArgument(roleType, "RoleType");
      checkNotNullArgument(user, "User");
      checkNotNullArgument(group, "Group");

      //TODO: does separate hasRelationship method in IdentityStore makes sense?

      Set<IdentityObjectRelationship> rels = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(group), createIdentityObject(user), ROLE);

      for (IdentityObjectRelationship rel : rels)
      {
         if (rel.getType().getName().equals(ROLE.getName()) && rel.getName() != null && rel.getName().equals(roleType.getName()))
         {
            return true;
         }
      }

      return false;
   }

   public boolean hasRole(String userName, String groupId, String roleTypeName) throws IdentityException
   {
      checkNotNullArgument(roleTypeName, "RoleType name");
      checkNotNullArgument(userName, "User name");
      checkNotNullArgument(groupId, "Group Id");

      User user = createUserFromId(userName);
      Group group = createGroupFromId(groupId);

      return hasRole(user, group, new SimpleRoleType(roleTypeName));
   }

   public Role getRole(RoleType roleType, User user, Group group) throws IdentityException
   {
      checkNotNullArgument(roleType, "RoleType");
      checkNotNullArgument(user, "User");
      checkNotNullArgument(group, "Group");

      //TODO: does separate hasRelationship method in IdentityStore makes sense?

      Set<IdentityObjectRelationship> rels = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(group), createIdentityObject(user), ROLE);

      if (rels.size() == 0)
      {
         return null;
      }

      for (IdentityObjectRelationship relationship : rels)
      {
         if (roleType.getName().equals(relationship.getName()))
         {
            return new SimpleRole(new SimpleRoleType(relationship.getName()),
               createUser(relationship.getToIdentityObject()),
               createGroup(relationship.getFromIdentityObject()));
         }
      }

      return null;


   }

   public Role getRole(String roleTypeName, String userId, String groupId) throws IdentityException
   {
      checkNotNullArgument(roleTypeName, "RoleType name");
      checkNotNullArgument(userId, "User id");
      checkNotNullArgument(groupId, "Group Id");

      // TODO: Check if relationship is present in the store

      User user = createUserFromId(userId);
      Group group = createGroupFromId(groupId);
      RoleType roleType = new SimpleRoleType(roleTypeName);

      return getRole(roleType, user, group);
   }

   public Collection<RoleType> findRoleTypes(User user, Group group) throws IdentityException
   {

      checkNotNullArgument(user, "User");
      checkNotNullArgument(group, "Group");

      return findRoleTypes(user, group, null);
   }

   public Collection<RoleType> findRoleTypes(User user, Group group, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(user, "User");
      checkNotNullArgument(group, "Group");

      Set<IdentityObjectRelationship> rels = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(group), createIdentityObject(user), ROLE);
      Set<RoleType> types = new HashSet<RoleType>();

      for (IdentityObjectRelationship rel : rels)
      {
         types.add(new SimpleRoleType(rel.getName()));
      }

      return types;


   }

   public Collection<RoleType> findRoleTypes(String userName, String groupId, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(userName, "User name");
      checkNotNullArgument(groupId, "Group Id");

      User user = createUserFromId(userName);
      Group group = createGroupFromId(groupId);

      return findRoleTypes(user, group, criteria);
   }

   public Collection<RoleType> findUserRoleTypes(User user) throws IdentityException
   {
      checkNotNullArgument(user, "User");

      return findUserRoleTypes(user, null);
   }

   public Collection<RoleType> findUserRoleTypes(User user, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(user,  "User");

      Set<RoleType> types = new HashSet<RoleType>();

      try
      {
         Collection<String> names = getRepository().getRelationshipNames(getInvocationContext(), createIdentityObject(user), convertSearchControls(criteria));

         for (String name : names)
         {
            types.add(new SimpleRoleType(name));
         }

         return types;

      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported", e);
      }

   }

   public Collection<RoleType> findUserRoleTypes(String userName, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(userName, "User name");

      User user = createUserFromId(userName);

      return findUserRoleTypes(user, criteria);
   }

   public Collection<RoleType> findGroupRoleTypes(Group group) throws IdentityException
   {
      checkNotNullArgument(group, "Group");

      return findGroupRoleTypes(group, null);
   }

   public Collection<RoleType> findGroupRoleTypes(String groupId, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(groupId, "Group Id");

      Group group = createGroupFromId(groupId);

      return findGroupRoleTypes(group, criteria);
   }

   public Collection<RoleType> findGroupRoleTypes(Group group, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(group, "Group");

      Set<RoleType> types = new HashSet<RoleType>();

      try
      {
         Collection<String> names = getRepository().getRelationshipNames(getInvocationContext(), createIdentityObject(group), convertSearchControls(criteria));

         for (String name : names)
         {
            types.add(new SimpleRoleType(name));
         }

         return types;

      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported");
      }

   }

   public Collection<User> findUsersWithRelatedRole(Group group, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(group, "Group");

      List<User> identities = new LinkedList<User>();


      Collection<IdentityObject> ios = null;

      ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(group), ROLE, true, convertSearchControls(criteria));

      for (IdentityObject io : ios)
      {
         identities.add(createUser(io));
      }

      return identities;
   }

   public Collection<User> findUsersWithRelatedRole(String groupId, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(groupId, "Group id");

      Group group = createGroupFromId(groupId);

      return findUsersWithRelatedRole(group, criteria);
   }

   public Collection<Group> findGroupsWithRelatedRole(User user, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(user, "User");

      List<Group> identities = new LinkedList<Group>();


      Collection<IdentityObject> ios = null;

      ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(user), ROLE, false, convertSearchControls(criteria));

      for (IdentityObject io : ios)
      {

         identities.add(createGroup(io));
      }

      return identities;
   }

   public Collection<Group> findGroupsWithRelatedRole(String userName, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(userName, "User name");

      User user = createUserFromId(userName);

      return findGroupsWithRelatedRole(user, criteria);
   }

   public Collection<Group> findGroupsWithRelatedRole(User user, String groupType, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(user, "User");
//      checkNotNullArgument(groupType, "Group type");

      List<Group> identities = new LinkedList<Group>();

      Collection<IdentityObject> ios = null;

      ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(user), ROLE, false, convertSearchControls(criteria));

      if (groupType != null)
      {
         IdentityObjectType iot = getIdentityObjectType(groupType);

         for (IdentityObject io : ios)
         {
            if (io.getIdentityType().getName().equals(iot.getName()))
            {
               identities.add(createGroup(io));
            }
         }
      }
      else
      {
         for (IdentityObject io : ios)
         {
            identities.add(createGroup(io));
         }
      }


      return identities;
   }

   public Collection<Group> findGroupsWithRelatedRole(String userName, String groupType, IdentitySearchCriteria criteria) throws IdentityException
   {
      checkNotNullArgument(userName, "User name");
//      checkNotNullArgument(groupType, "Group type");


      User user = createUserFromId(userName);

      return findGroupsWithRelatedRole(user, groupType, criteria);
   }

   public Collection<Role> findRoles(IdentityType identityType, RoleType roleType) throws IdentityException
   {
      checkNotNullArgument(identityType, "IdentityType");
      //checkNotNullArgument(roleType, "RoleType");

      Set<Role> roles = new HashSet<Role>();

      Set<IdentityObjectRelationship> relationships = null;

      // If Identity then search for parent relationships
      if (identityType instanceof User)
      {
         relationships = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(identityType), ROLE, false, true, null);
      }
      // If Group then search for child relationships
      else
      {
         relationships = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(identityType), ROLE, true, true, null);
      }

      for (IdentityObjectRelationship relationship : relationships)
      {
         if (roleType != null)
         {
            if (roleType.getName().equals(relationship.getName()))
            {
               roles.add(new SimpleRole(new SimpleRoleType(relationship.getName()), createUser(relationship.getToIdentityObject()), createGroup(relationship.getFromIdentityObject())));
            }
         }
         else
         {
            roles.add(new SimpleRole(new SimpleRoleType(relationship.getName()), createUser(relationship.getToIdentityObject()), createGroup(relationship.getFromIdentityObject())));
         }
      }

      return roles;

   }

   public Collection<Role> findRoles(String id, String roleTypeName) throws IdentityException
   {
      checkNotNullArgument(id, "Group id or User name");
//      checkNotNullArgument(roleTypeName, "RoleType name");

      RoleType roleType = roleTypeName != null ? new SimpleRoleType(roleTypeName) : null;
      return findRoles(createIdentityTypeFromId(id), roleType);
   }

   public Map<String, String> getProperties(RoleType roleType)  throws IdentityException
   {
      checkNotNullArgument(roleType, "RoleType name");
      
      return getRepository().getRelationshipNameProperties(getInvocationContext(), roleType.getName());
   }

   public Map<String, String> getProperties(String roleTypeName)  throws IdentityException
   {
      checkNotNullArgument(roleTypeName, "RoleType name");

      return getProperties(new SimpleRoleType(roleTypeName));
   }

   public Map<String, String> getProperties(Role role) throws IdentityException
   {
      checkNotNullArgument(role, "Role");

      return getRepository().getRelationshipProperties(getInvocationContext(), createIdentityObjectRelationship(role));
   }

   public void setProperty(Role role, String name, String value) throws IdentityException
   {
      checkNotNullArgument(role, "Role");
      checkNotNullArgument(name, "Property name");
      checkNotNullArgument(value, "Property value");

      Map<String, String> props = new HashMap<String, String>();
      props.put(name, value);


      getRepository().setRelationshipProperties(getInvocationContext(), createIdentityObjectRelationship(role), props);
   }

   public void setProperty(RoleType roleType, String name, String value) throws IdentityException
   {
      checkNotNullArgument(roleType, "RoleType");
      checkNotNullArgument(name, "Property name");
      checkNotNullArgument(value, "Property value");

      Map<String, String> props = new HashMap<String, String>();
      props.put(name, value);


      getRepository().setRelationshipNameProperties(getInvocationContext(), roleType.getName(), props);
   }

   public void setProperty(String roleTypeName, String name, String value) throws IdentityException
   {
      checkNotNullArgument(roleTypeName, "RoleType name");
      checkNotNullArgument(name, "Property name");
      checkNotNullArgument(value, "Property value");

      setProperty(new SimpleRoleType(roleTypeName), name, value);
   }

   public void setProperties(Role role, Map<String, String> properties) throws IdentityException
   {
      checkNotNullArgument(role, "Role");
      checkNotNullArgument(properties, "Properties");


      getRepository().setRelationshipProperties(getInvocationContext(), createIdentityObjectRelationship(role), properties);
   }

   public void setProperties(RoleType roleType, Map<String, String> properties) throws IdentityException
   {
      checkNotNullArgument(roleType, "RoleType");
      checkNotNullArgument(properties, "Properties");


      getRepository().setRelationshipNameProperties(getInvocationContext(), roleType.getName(), properties);
   }

   public void setProperties(String roleTypeName, Map<String, String> properties) throws IdentityException
   {
      checkNotNullArgument(roleTypeName, "RoleType name");

      setProperties(new SimpleRoleType(roleTypeName), properties);
   }

   public void removeProperty(Role role, String name) throws IdentityException
   {
      checkNotNullArgument(role, "Role");
      checkNotNullArgument(name, "Property name");

      Set<String> names = new HashSet<String>();
      names.add(name);

      getRepository().removeRelationshipProperties(getInvocationContext(), createIdentityObjectRelationship(role), names);
   }

   public void removeProperty(RoleType roleType, String name) throws IdentityException
   {
      checkNotNullArgument(roleType, "RoleType");
      checkNotNullArgument(name, "Property name");

      Set<String> names = new HashSet<String>();
      names.add(name);


      getRepository().removeRelationshipNameProperties(getInvocationContext(), roleType.getName(), names);
   }

   public void removeProperty(String roleTypeName, String name) throws IdentityException
   {
      checkNotNullArgument(roleTypeName, "RoleType name");

      removeProperty(new SimpleRoleType(roleTypeName), name);
   }
}
