/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.cache;

import org.jboss.identity.idm.spi.store.AttributeStore;
import org.jboss.identity.idm.spi.store.IdentityStoreInvocationContext;
import org.jboss.identity.idm.spi.store.IdentityStoreSession;
import org.jboss.identity.idm.spi.model.IdentityObjectType;
import org.jboss.identity.idm.spi.model.IdentityObjectAttribute;
import org.jboss.identity.idm.spi.model.IdentityObject;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityObjectAttributeMetaData;
import org.jboss.identity.idm.spi.cache.IdentityStoreCacheProvider;
import org.jboss.identity.idm.common.exception.IdentityException;

import java.util.logging.Logger;
import java.util.Set;
import java.util.Map;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class JBossCacheAttributeStoreWrapper implements AttributeStore
{

   private static Logger log = Logger.getLogger(JBossCacheAttributeStoreWrapper.class.getName());

   private final AttributeStore attributeStore;

   protected final IdentityStoreCacheProvider cacheSupport;

   // if update/add/remove operation on attributes should result in getAttributes() query to store new set in cache
   // with many updates it can add additional cost
   private boolean reloadAttributesToCacheOnUpdate = true;


   public JBossCacheAttributeStoreWrapper(AttributeStore attributeStore, IdentityStoreCacheProvider cacheSupport)
   {
      //cacheSupport = new IdentityStoreCacheSupportImpl(cacheConfigInputStream);
      this.cacheSupport = cacheSupport;
      this.attributeStore = attributeStore;
   }

   public String getId()
   {
      return attributeStore.getId();
   }

   public IdentityStoreSession createIdentityStoreSession() throws IdentityException
   {
      return attributeStore.createIdentityStoreSession();
   }

   public Set<String> getSupportedAttributeNames(IdentityStoreInvocationContext invocationContext,
                                                 IdentityObjectType identityType) throws IdentityException
   {
      // Doesn't need to be cached
      return attributeStore.getSupportedAttributeNames(invocationContext, identityType);
   }

   public Map<String, IdentityObjectAttributeMetaData> getAttributesMetaData(IdentityStoreInvocationContext invocationContext,
                                                                             IdentityObjectType identityType)
   {
      // Doesn't need to be cached
      return attributeStore.getAttributesMetaData(invocationContext, identityType);
   }

   public Map<String, IdentityObjectAttribute> getAttributes(IdentityStoreInvocationContext invocationContext,
                                                             IdentityObject identity) throws IdentityException
   {
      Map<String, IdentityObjectAttribute> results = cacheSupport.getAttributesFromCache(identity.getName(), identity.getIdentityType());

      if (results == null)
      {
         results = attributeStore.getAttributes(invocationContext, identity);

         if (results != null && results.size() > 0)
         {
            cacheSupport.putIntoCache(identity, results);
         }
      }

      return results;
   }

   public IdentityObjectAttribute getAttribute(IdentityStoreInvocationContext invocationContext,
                                               IdentityObject identity,
                                               String name) throws IdentityException
   {
      Map<String, IdentityObjectAttribute> results = cacheSupport.getAttributesFromCache(identity.getName(), identity.getIdentityType());

      if (results != null && results.containsKey(name))
      {
         return results.get(name);
      }
      else
      {
         return attributeStore.getAttribute(invocationContext, identity, name);

         //TODO: add this attribute to the node map
      }

   }

   public void updateAttributes(IdentityStoreInvocationContext invocationCtx,
                                IdentityObject identity,
                                IdentityObjectAttribute[] attributes) throws IdentityException
   {
      attributeStore.updateAttributes(invocationCtx, identity, attributes);

      cacheSupport.removeAttributesFromCache(identity);

      if (reloadAttributesToCacheOnUpdate)
      {
         Map<String, IdentityObjectAttribute> results = attributeStore.getAttributes(invocationCtx, identity);

         if (results != null && results.size() > 0)
         {
            cacheSupport.putIntoCache(identity, results);
         }
      }
      else
      {
         // update attributes in node map directly
      }

   }

   public void addAttributes(IdentityStoreInvocationContext invocationCtx,
                             IdentityObject identity,
                             IdentityObjectAttribute[] attributes) throws IdentityException
   {
      attributeStore.addAttributes(invocationCtx, identity, attributes);

      cacheSupport.removeAttributesFromCache(identity);

      if (reloadAttributesToCacheOnUpdate)
      {
         Map<String, IdentityObjectAttribute> results = attributeStore.getAttributes(invocationCtx, identity);

         if (results != null && results.size() > 0)
         {
            cacheSupport.putIntoCache(identity, results);
         }
      }
      else
      {
         // add attributes in node map directly
      }
   }

   public void removeAttributes(IdentityStoreInvocationContext invocationCtx,
                                IdentityObject identity,
                                String[] attributeNames) throws IdentityException
   {
      attributeStore.removeAttributes(invocationCtx, identity, attributeNames);

      cacheSupport.removeAttributesFromCache(identity);

      if (reloadAttributesToCacheOnUpdate)
      {
         Map<String, IdentityObjectAttribute> results = attributeStore.getAttributes(invocationCtx, identity);

         if (results != null && results.size() > 0)
         {
            cacheSupport.putIntoCache(identity, results);
         }
      }
      else
      {
         // remove attributes in node map directly
      }
   }

   protected Logger getLog()
   {
      return log;
   }

//   protected Cache getCache()
//   {
//      return cache;
//   }

   @Override
   public String toString()
   {
      return "JBossCacheAttributeStoreWrapper (AttributeStore=" + attributeStore.getId() + ")";
   }

}
