/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist;

import java.util.Collection;
import java.util.Map;

import javassist.CtClass;
import javassist.bytecode.SignatureAttribute.ClassSignature;
import javassist.bytecode.SignatureAttribute.TypeArgument;

import org.jboss.reflect.plugins.ClassInfoImpl;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.DelegateClassInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.reflect.spi.TypeInfoFactory;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class JavassistParameterizedClassInfo extends DelegateClassInfo
{

   /** The serialVersionUID */
   private static final long serialVersionUID = 2;
   
   /** The factory */
   private final JavassistTypeInfoFactoryImpl factory;
   
   private final TypeArgument[] typeArguments;
   
   private volatile TypeInfo[] typeArgumentInfos = ClassInfoImpl.UNKNOWN_TYPES; 
   
   JavassistParameterizedClassInfo(JavassistTypeInfoFactoryImpl factory, ClassInfo delegate, TypeArgument[] typeArguments)
   {
      super(delegate);
      this.factory = factory;
      this.typeArguments = typeArguments;
   }
   
   public JavassistParameterizedClassInfo(JavassistTypeInfoFactoryImpl factory, ClassInfo delegate, TypeInfo[] typeArgumentInfos)
   {
      this(factory, delegate, (TypeArgument[])null);
      this.typeArgumentInfos = typeArgumentInfos;
   }
   
   @Override
   public TypeInfoFactory getTypeInfoFactory()
   {
      return factory;
   }

   @Override
   public TypeInfo[] getActualTypeArguments()
   {
      if (typeArgumentInfos == ClassInfoImpl.UNKNOWN_TYPES && typeArguments.length > 0)
      {
         try
         {
            TypeInfo[] infos = new TypeInfo[typeArguments.length];
            for (int i = 0 ; i < typeArguments.length ; i++)
            {
               infos[i] = (TypeInfo)factory.get(typeArguments[i].getType().toString(), delegate.getClassLoader());
            }
            typeArgumentInfos = infos;
         }
         catch (ClassNotFoundException e)
         {
            // AutoGenerated
            throw new RuntimeException(e);
         }
      }
         
      return typeArgumentInfos;
   }

   @Override
   public TypeInfo getComponentType()
   {
      return findTypeInfo(Collection.class.getName(), 0, CollectionTypeChecker.INSTANCE);
   }

   @Override
   public TypeInfo getKeyType()
   {
      return findTypeInfo(Map.class.getName(), 0, MapTypeChecker.INSTANCE);
   }

   @Override
   public TypeInfo getValueType()
   {
      return findTypeInfo(Map.class.getName(), 1, MapTypeChecker.INSTANCE);
   }
   
   private TypeInfo findTypeInfo(String target, int parameter, TypeChecker checker)
   {
      ClassSignature sig = JavassistHelper.getClassSignature(((JavassistTypeInfo)delegate).getCtClass());
      if (sig == null)
         return delegate.getComponentType();
      
      if (!checker.check(this))
         return null;

      try
      {
         CtClass ctClass = ((JavassistTypeInfo)delegate).getCtClass();
         CtClass collection = ctClass.getClassPool().get(target);
         int index = JavassistHelper.determineInfoIndex(getActualTypeArguments(), ((JavassistTypeInfo)delegate).getCtClass(), collection, parameter);
         return getActualTypeArguments()[index];
      }
      catch (Exception e1)
      {
         throw new RuntimeException(e1);
      }
   }

   private static interface TypeChecker
   {
      boolean check(JavassistParameterizedClassInfo info);
   }
   
   private static class MapTypeChecker implements TypeChecker
   {
      final static MapTypeChecker INSTANCE = new MapTypeChecker();

      public boolean check(JavassistParameterizedClassInfo info)
      {
         return info.isMap();
      }
   }
   
   private static class CollectionTypeChecker implements TypeChecker
   {
      final static CollectionTypeChecker INSTANCE = new CollectionTypeChecker();

      public boolean check(JavassistParameterizedClassInfo info)
      {
         return info.isCollection();
      }
   }
}
