/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.javassist;

import java.util.Arrays;

import javassist.CtClass;
import javassist.CtMethod;
import javassist.NotFoundException;
import javassist.bytecode.SignatureAttribute.BaseType;
import javassist.bytecode.SignatureAttribute.MethodSignature;

import org.jboss.reflect.plugins.AnnotationHelper;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.MethodInfo;
import org.jboss.reflect.spi.MutableMethodInfo;
import org.jboss.reflect.spi.MutableMethodInfoCommand;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.util.JBossStringBuilder;


/**
 * MethodInfo that relies on Javassist to answer reflective queries and to invoke
 * the represented method.
 * 
 *  This class also relies on Javassist to perform the instrumentation operations defined in
 *  {@code MutableMethodInfo}.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 104251 $
 * @see MutableMethodInfo
 */
// TODO: review hierarchy structure. Shouldn't JavassistMethodInfo extend MethodInfoImpl, just like
//ReflectMethodInfoImpl?
public class JavassistMethodInfo extends JavassistAnnotatedParameterInfo implements MutableMethodInfo
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 101183748227690112L;

   /** The method implementation */
   private transient volatile JavassistMethod method;
   
   /** The return type */
   private transient TypeInfo returnType;
   
   /**
    * Create a new JavassistMethodInfo.
    * 
    * @param annotationHelper the annotation helper
    * @param typeInfo the type info
    * @param ctMethod the method
    */
   public JavassistMethodInfo(AnnotationHelper annotationHelper, JavassistTypeInfo typeInfo, CtMethod ctMethod)
   {
      super(annotationHelper, typeInfo, ctMethod);
      //this.ctMethod = ctMethod;
   }

   public String getName()
   {
      return ((CtMethod)ctBehavior).getName();
   }

   public ClassInfo getDeclaringClass()
   {
      return typeInfo;
   }

   public TypeInfo getReturnType()
   {
      if (returnType != null)
         return returnType;
      try
      {
         CtMethod ctMethod = (CtMethod)ctBehavior;
         MethodSignature sig = JavassistHelper.getMethodSignature(ctMethod);
         if (sig != null)
         {
            if (sig.getReturnType() instanceof BaseType == false)
               return typeInfo.getFactory().getTypeInfo(typeInfo.getClassLoaderInternal(), sig.getReturnType(), JavassistTypeVariableSpy.createForBehavior(ctMethod, sig));
         }

         CtClass clazz = ctMethod.getReturnType();
         return typeInfo.getFactory().getTypeInfo(clazz);
      }
      catch (NotFoundException e)
      {
         throw JavassistTypeInfoFactoryImpl.raiseClassNotFound("for return type of method " + getName(), e);
      }
   }

   public Object invoke(Object target, Object[] args) throws Throwable
   {
      if (method == null)
         method = JavassistReflectionFactory.INSTANCE.createMethod(this);
      
      JavassistAccessController.checkAccess(this);
      
      return method.invoke(target, args);
   }

   @Override
   protected int getHashCode()
   {
      return getName().hashCode();
   }

   @Override
   public boolean equals(Object obj)
   {
      if (obj == this)
         return true;
      if (obj == null || obj instanceof MethodInfo == false)
         return false;

      final MethodInfo other = (MethodInfo) obj;

      if (getName().equals(other.getName()) == false)
         return false;
//      if (getDeclaringClass().equals(other.getDeclaringClass()) == false)
//         return false;
//      if (getReturnType().equals(other.getReturnType()) == false)
//         return false;
      return Arrays.equals(getParameterTypes(), other.getParameterTypes());
   }

   @Override
   public void toShortString(JBossStringBuilder buffer)
   {
      buffer.append(getName());
   }

   @Override
   protected void toString(JBossStringBuilder buffer)
   {
      buffer.append("name=").append(getName());
      super.toString(buffer);
   }

   //TODO: need to be implemented...
   public void executeCommand(MutableMethodInfoCommand mmc)
   {
   }
   
   public CtMethod getCtMethod()
   {
      return (CtMethod)ctBehavior;
   }

   public void setName(String name)
   {
      ((CtMethod)ctBehavior).setName(name);
      typeInfo.clearMethodCache();
   }

   public void setReturnType(String returnType)
   {
      throw new RuntimeException("Method not supported by Javassist");
   }

   public void setReturnType(ClassInfo returnType)
   {
      throw new RuntimeException("Method not supported by Javassist");
   }
}
