/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist.bytecode;

import java.lang.reflect.Method;

import javassist.Modifier;
import javassist.bytecode.MethodInfo;
import javassist.util.proxy.RuntimeSupport;

import org.jboss.reflect.plugins.javassist.JavassistConstructor;
import org.jboss.reflect.plugins.javassist.JavassistConstructorInfo;

/**
 * <p>Class to create implementations of the {{@link JavassistConstructor} interface.</p>
 * 
 * <p>This implementation generates raw bytecode to avoid the overhead of compilation via javassist. If 
 * <code>sun.reflect.MagicAccessorImpl</code> is used as the <code>superClass</code> field the implementation
 * classes can access private and protected members of the target class.</p>
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
class JavassistConstructorFactory extends JavassistBehaviorFactory
{
   /** The names of the interfaces we are implementing */ 
   private static final String[] interfaceNames = new String[] {JavassistConstructor.class.getName()};
   
   /** The methods from the interface that are being implemented */ 
   private static final Method[] methods = new Method[1];
   
   /** The descriptors for the methods */
   private static final String[] methodDescriptors = new String[1];
   
   static
   {
      try
      {
         methods[0] = SecurityActions.getDeclaredMethod(JavassistConstructor.class, "newInstance", Object[].class);
      }
      catch (NoSuchMethodException e)
      {
         throw new RuntimeException(e);
      }
      methodDescriptors[0] = RuntimeSupport.makeDescriptor(methods[0]); 
   }

   /** The name of the class being generated */
   private final String className = JavassistConstructor.class.getName() + counter.incrementAndGet();
   
   /**
    * Constructor
    * 
    * @param superClass the super class to use for the implementation
    * @param ctConstructor the constructor we are generating a {@link JavassistConstructor} for
    * @param debug true to cause the class bytes to be output to the file system so they can be inspected with a decompiler/javap
    */      
   JavassistConstructorFactory(Class<?> superClass, JavassistConstructorInfo constructor, boolean debug)
   {
      super(superClass, constructor, debug);
   }

   @Override
   String getGeneratedClassName()
   {
      return className;
   }
   
   @Override
   String[] getInterfaceNames()
   {
      return interfaceNames;
   }

   @Override
   boolean implementMethod(int index, ClassFileWriterContext<?> cfwc)
   {
      if (index >= methods.length)
         return false;

      Method method = methods[index]; 

      cfwc.beginMethod(Modifier.PUBLIC, method.getName(), methodDescriptors[index], THROWABLE_EXCEPTIONS);
      
      cfwc.addNew(ClassFileWriterContext.jvmClassName(getBehavior().getDeclaringClass().getName()));
      cfwc.addDup();
      
      addParameters(cfwc, getBehavior().getSignatureKey(), 1);
      
      cfwc.addInvokeSpecial(ClassFileWriterContext.jvmClassName(getBehavior().getDeclaringClass().getName()), MethodInfo.nameInit, getBehavior().getDescriptor());
      cfwc.addAReturn();
      
      //We need 3 local variable slots.
      //One for 'this', one for the target reference and one for the argument array.
      //These are all object references and so take one slot each
      
      //TODO the max stack
      cfwc.endMethod(3);
      
      return true;
   }
}