/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.reflect.spi;

/**
 * A MutableClassInfo.
 * 
 * @author <a href="stale.pedersen@jboss.org">Stale W. Pedersen</a>
 * @version $Revision: 1.1 $
 */
public interface MutableClassInfo extends ClassInfo
{
   /**
    * Get the declared method without parameters.
    * 
    * @param name the method name
    * @return the method info
    */
   MutableMethodInfo getDeclaredMethod(String name);
   
   /**
    * Get the declared method
    * 
    * @param name the method name
    * @param parameters the parameters
    * @return the method info
    */
   MutableMethodInfo getDeclaredMethod(String name, TypeInfo... parameters);
   
   /**
    * Get the declared method
    * Note: the parameters will be loaded with the classloader defined in
    * TypeInfoFactory.
    * 
    * @param name the method name
    * @param parameters the parameters
    * @return the method info
    * @throws ClassNotFoundException 
    */
   MutableMethodInfo getDeclaredMethod(String name, String... parameters) throws ClassNotFoundException;
   
   /**
    * Get the declared methods
    * 
    * @return the methods
    */
   MutableMethodInfo[] getDeclaredMethods();
   
   /**
    * Get the declared constructors
    * 
    * @return the constructors
    */
   MutableConstructorInfo[] getDeclaredConstructors();

   /**
    * Gets the default constructor (parameterless)
    * 
    * @return the constructor
    */
   MutableConstructorInfo getDeclaredConstructor();
   
   /**
    * Get a declared constructor
    * 
    * @param parameters the parameters
    * @return the constructor
    */
   MutableConstructorInfo getDeclaredConstructor(TypeInfo... parameters);

   /**
    * Get a declared constructor
    * Note: the parameters will be loaded with the classloader defined in
    * TypeInfoFactory.
    * 
    * @param parameters the parameters
    * @return the constructor
    * @throws ClassNotFoundException 
    */
   MutableConstructorInfo getDeclaredConstructor(String... parameters) throws ClassNotFoundException;
   
   /**
    * Get the declared field
    * 
    * @param name the field name
    * @return the field
    */
   MutableFieldInfo getDeclaredField(String name);

   /**
    * Get the declared fields
    * 
    * @return the fields
    */
   MutableFieldInfo[] getDeclaredFields();
   
   /**
    * Compiles the code included in Body and returns a MutableMethodInfo representation of it.
    * The Body must include the whole declaration of the method.
    * 
    * @param body  contains the entire declaration of the method, including its signature
    * @return      the created method
    */
   MutableMethodInfo createMutableMethod(Body body);
   
   /**
    * Compiles an empty method with the signature given by the parameters.
    * 
    * @param modifiers   the modifiers of the method to be created
    * @param returnType  the return type of the method to be created
    * @param name        the name of the method to be created
    * @param parameters  the parameters of the method to be created
    * @param exceptions  the exceptions that the method to be created declares to throw
    * @return            the created method
    */
   MutableMethodInfo createMutableMethod(int modifiers, String returnType, 
         String name, String[] parameters, String[] exceptions);
   
   /**
    * Compiles an empty method with the signature given by the parameters.
    * 
    * @param modifiers   the modifiers of the method to be created
    * @param returnType  the return type of the method to be created
    * @param name        the name of the method to be created
    * @param parameters  the parameters of the method to be created
    * @param exceptions  the exceptions that the method to be created declares to throw
    * @return            the created method
    */
   MutableMethodInfo createMutableMethod(int modifiers, ClassInfo returnType, 
         String name, ClassInfo[] parameters, ClassInfo[] exceptions);
   
   
   /**
    * Compiles a method with the signature and body given by the parameters.
    * Note that the source code of the body must be surrounded by <code>{}</code>.
    * 
    * @param modifiers   the modifiers of the method to be created
    * @param returnType  the return type of the method to be created
    * @param name        the name of the method to be created
    * @param body        the body of the method to be created
    * @param parameters  the parameters of the method to be created
    * @param exceptions  the exceptions that the method to be created declares to throw
    * @return            the created method
    */
   MutableMethodInfo createMutableMethod(int modifiers, String returnType, String name, 
         Body body, String[] parameters, String[] exceptions);

 
   /**
    * Compiles a method with the signature and body given by the parameters.
    * Note that the source code of the body must be surrounded by <code>{}</code>.
    * 
    * @param modifiers   the modifiers of the method to be created
    * @param returnType  the return type of the method to be created
    * @param name        the name of the method to be created
    * @param body        the body of the method to be created
    * @param parameters  the parameters of the method to be created
    * @param exceptions  the exceptions that the method to be created declares to throw
    * @return            the created method
    */
   MutableMethodInfo createMutableMethod(int modifiers, ClassInfo returnType, String name,
         Body body, ClassInfo[] parameters, ClassInfo[] exceptions);

   
   /**
    * Compiles the code included in the Body parameter and returns a MutableConstructorInfo
    * representation of it. The Body must include the whole declaration.
    * The 
    * 
    * @param body contains the code of the constructor to be created
    * @return     the created constructor
    */
   MutableConstructorInfo createMutableConstructor(Body body);
   
   /**
    * Creates an empty constructor with parameters given.
    * 
    * @param modifiers   the modifiers of the constructor to be created
    * @param parameters  the parameters of the constructor to be created
    * @param exceptions  the exceptions that the constructor to be created declares to throw
    * @return            the created constructor
    */
   MutableConstructorInfo createMutableConstructor(int modifiers, String[] parameters, 
         String[] exceptions);
   
   /**
    * Creates an empty constructor with the parameters given.
    * 
    * @param modifiers   the modifiers of the constructor to be created
    * @param parameters  the parameters of the constructor to be created
    * @param exceptions  the exceptions that the constructor to be created declares to throw
    * @return            the created constructor
    */
   MutableConstructorInfo createMutableConstructor(int modifiers, ClassInfo[] parameters, 
         ClassInfo[] exceptions);
   
   /**
    * Creates a constructor with the parameters given.
    * Note that the source text of the Body must be surrounded by <code>{}</code>.
    * 
    * @param modifiers   the modifiers of the constructor to be created
    * @param body        the body of the constructor to be created
    * @param parameters  the parameters of the constructor to be created
    * @param exceptions  the exceptions that the constructor to be created declares to throw
    * @return            the created constructor
    */
   MutableConstructorInfo createMutableConstructor(int modifiers, Body body,
         String[] parameters, String[] exceptions);
   
   /**
    * Creates a constructor with the parameters given.
    * Note that the source text of the Body must be surrounded by <code>{}</code>.
    * 
    * @param modifiers   the modifiers of the constructor to be created
    * @param body        the body of the constructor to be created
    * @param parameters  the parameters of the constructor to be created
    * @param exceptions  the exceptions that the constructor to be created declares to throw
    * @return            the created constructor
    */
   MutableConstructorInfo createMutableConstructor(int modifiers, Body body,
         ClassInfo[] parameters, ClassInfo[] exceptions);
   
   /**
    * Creates a field connected to this class with the parameters given.
    * 
    * @param modifiers  the modifiers of the field to be created
    * @param type       the type of the field to be created
    * @param name       the name of the field to be created
    * @return           the created field
    */
   MutableFieldInfo createMutableField(int modifiers, String type, String name);
   
   /**
    * Creates a field connected to this class with the parameters given.
    * 
    * @param modifiers  the modifiers of the field to be created
    * @param type       the type of the field to be created
    * @param name       the name of the field to be created
    * @return           the created field
    */
   MutableFieldInfo createMutableField(int modifiers, ClassInfo type, String name);
   
   /**
    * Add a method
    * 
    * @param mmi
    */
   void addMethod(MutableMethodInfo mmi);
   
   /**
    * Remove the specified method from this class
    * 
    * @param mmi
    */
   void removeMethod(MutableMethodInfo mmi);
   
   /**
    * Add a constructor
    * 
    * @param mci
    */
   void addConstructor(MutableConstructorInfo mci);
   
   /**
    * Remove the specified constructor from this class
    * 
    * @param mci
    */
   void removeConstructor(MutableConstructorInfo mci);
   
   /**
    * Add a field
    * 
    * @param mfi
    */
   void addField(MutableFieldInfo mfi);
   
   /**
    * Remove a field
    * 
    * @param mfi
    */
   void removeField(MutableFieldInfo mfi);
   
   /**
    * Converts the class to a Class file. 
    * After this method is called, no modifications to the class are allowed.
    * 
    * @return this class in bytecodes
    */
   byte[] toByteCode();
   
}
