/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.javassist;

import java.lang.annotation.Annotation;
import java.lang.ref.WeakReference;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.reflect.plugins.*;
import org.jboss.reflect.plugins.introspection.IntrospectionTypeInfoFactory;
import org.jboss.reflect.plugins.javassist.classpool.ClassPoolFactory;
import org.jboss.reflect.plugins.javassist.classpool.DefaultClassPoolFactory;
import org.jboss.reflect.spi.*;
import org.jboss.util.JBossStringBuilder;
import org.jboss.util.collection.WeakClassCache;

import javassist.*;
import javassist.NotFoundException;
import javassist.bytecode.Descriptor;
import javassist.bytecode.SignatureAttribute;
import javassist.bytecode.SignatureAttribute.ArrayType;
import javassist.bytecode.SignatureAttribute.BaseType;
import javassist.bytecode.SignatureAttribute.ClassType;
import javassist.bytecode.SignatureAttribute.ObjectType;
import javassist.bytecode.SignatureAttribute.TypeArgument;

/**
 * A javassist type factory.
 * TODO: need to fix the cl stuff
 *
 * @author <a href="mailto:adrian@jboss.org">Adrian Brock</a>
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 *
 * @version $Revision: 110312 $
 */
public class JavassistTypeInfoFactoryImpl extends WeakClassCache<TypeInfo> implements MutableTypeInfoFactory, AnnotationHelper
{
   private static ClassPoolFactory poolFactory = DefaultClassPoolFactory.getInstance();

   /** Tmp invocation results */
   private static ThreadLocal<Map<CtClass, TypeInfo>> results = new ThreadLocal<Map<CtClass, TypeInfo>>();

   JavassistTypeInfoFactoryImpl()
   {
   }

   public static ClassPoolFactory getPoolFactory()
   {
      return poolFactory;
   }

   public static void setPoolFactory(ClassPoolFactory classPoolFactory)
   {
      poolFactory = classPoolFactory;
   }

   static final AnnotationValue[] NO_ANNOTATIONS = new AnnotationValue[0];
   /**
    * Raise NoClassDefFoundError for javassist not found
    *
    * @param name the name
    * @param e the not found error
    * @return never
    * @throws NoClassDefFoundError always
    */
   public static NoClassDefFoundError raiseClassNotFound(String name, NotFoundException e) throws NoClassDefFoundError
   {
      NoClassDefFoundError ex = new NoClassDefFoundError("Unable to find class " + name);
      if (e.getCause() != null)
         ex.initCause(e.getCause()); // Hide the javassist error
      throw ex;
   }

   /**
    * Raise NoClassDefFoundError for javassist not found
    *
    * @param name the name
    * @param e the not found error
    * @return never
    * @throws NoClassDefFoundError always
    */
   public static NoClassDefFoundError raiseClassNotFound(String name, ClassNotFoundException e) throws NoClassDefFoundError
   {
      NoClassDefFoundError ex = new NoClassDefFoundError("Unable to find class " + name);
      ex.initCause(e);
      throw ex;
   }

   /**
    * Raise NoClassDefFoundError for javassist not found
    *
    * @param name the name
    * @param e the not found error
    * @return never
    * @throws NoClassDefFoundError always
    */
   public static NoClassDefFoundError raiseMethodNotFound(String name, NotFoundException e) throws NoClassDefFoundError
   {
      NoSuchMethodError ex = new NoSuchMethodError("Unable to find method " + name);
      if (e.getCause() != null)
         ex.initCause(e.getCause()); // Hide the javassist error
      throw ex;
   }

   /**
    * Raise NoClassDefFoundError for javassist not found
    *
    * @param name the name
    * @param e the not found error
    * @return never
    * @throws NoClassDefFoundError always
    */
   public static NoClassDefFoundError raiseFieldNotFound(String name, NotFoundException e) throws NoClassDefFoundError
   {
      NoSuchFieldError ex = new NoSuchFieldError("Unable to find field " + name);
      if (e.getCause() != null)
         ex.initCause(e.getCause()); // Hide the javassist error
      throw ex;
   }

   @Override
   @SuppressWarnings("unchecked")
   protected TypeInfo instantiate(Class clazz)
   {
      CtClass ctClass = getCtClass(clazz.getName(), clazz.getClassLoader());
      return instantiate(ctClass, clazz);
   }

   protected TypeInfo instantiate(CtClass ctClass, Class<?> clazz)
   {
      boolean start = false;
      Map<CtClass, TypeInfo> tmp = results.get();
      if (tmp == null)
      {
         start = true;
         tmp = new HashMap<CtClass, TypeInfo>();
         results.set(tmp);
      }

      // if we return here, it means we're already in a loop,
      // hence no need to cleanup
      TypeInfo cached = tmp.get(ctClass);
      if (cached != null)
         return cached;

      try
      {
         if (ctClass.isArray())
         {
            TypeInfo componentType = getTypeInfo(ctClass.getComponentType());
            TypeInfo result = new JavassistArrayInfoImpl(this, ctClass, clazz, componentType);
            tmp.put(ctClass, result);
            return result;
         }
         else if (ctClass.isAnnotation())
         {
            JavassistAnnotationInfo result = new JavassistAnnotationInfo(this, ctClass, clazz);
            tmp.put(ctClass, result);

            CtMethod[] methods = ctClass.getDeclaredMethods();
            AnnotationAttributeImpl[] atttributes = new AnnotationAttributeImpl[methods.length];
            for (int i = 0 ; i < methods.length ; i++)
            {
               atttributes[i] = new AnnotationAttributeImpl(methods[i].getName(), getTypeInfo(methods[i].getReturnType()), null);
            }
            result.setAttributes(atttributes);

            return result;
         }
         else if (ctClass.isEnum())
         {
            JavassistEnumInfo enumInfo = new JavassistEnumInfo(this, ctClass, clazz);
            tmp.put(ctClass, enumInfo);

            CtField[] fields = ctClass.getFields();
            List<EnumConstantInfoImpl> constants = new ArrayList<EnumConstantInfoImpl>();
            for (CtField field : fields)
            {
               if (Modifier.isEnum(field.getModifiers()))
               {
                  AnnotationValue[] annotations = getAnnotations(field);
                  constants.add(new EnumConstantInfoImpl(field.getName(), enumInfo, annotations));
               }
            }
            enumInfo.setEnumConstants(constants.toArray(new EnumConstantInfoImpl[constants.size()]));

            return enumInfo;
         }
         else
         {
            TypeInfo result = new JavassistTypeInfo(this, ctClass, clazz);
            tmp.put(ctClass, result);
            return result;
         }
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException(e);
      }
      finally
      {
         if (start)
            results.remove();
      }
   }

   /**
    * Get the type info
    *
    * @param ctClass the ctClass
    * @return the typeinfo
    */
   protected TypeInfo getTypeInfo(CtClass ctClass)
   {
      try
      {
         String name = convertName(ctClass);
         ClassLoader cl = JavassistUtil.getClassLoader(ctClass);
         return getTypeInfo(name, cl);
      }
      catch (ClassNotFoundException e)
      {
         throw raiseClassNotFound(ctClass.getName(), e);
      }
   }

   /**
    * Convert name
    *
    * @param clazz the class
    * @return the converted name
    */
   protected static String convertName(CtClass clazz)
   {
      CtClass temp = clazz;
      if (temp.isArray())
      {
         JBossStringBuilder buffer = new JBossStringBuilder();
         try
         {
            while (temp.isArray())
            {
               buffer.append('[');
               temp = temp.getComponentType();
            }
            if (temp.isPrimitive())
            {
               CtPrimitiveType primitive = (CtPrimitiveType) temp;
               buffer.append(Character.toString(primitive.getDescriptor()));
            }
            else
            {
               buffer.append('L');
               buffer.append(temp.getName());
               buffer.append(';');
            }
            return buffer.toString();
         }
         catch (NotFoundException e)
         {
            throw raiseClassNotFound(clazz.getName(), e);
         }
      }
      return clazz.getName();
   }

   /**
    * Gets the CtClass corresponding to {@code name}.
    *
    * @param name the name of the CtClass to be retrieved
    * @param classLoader the classLoader that loaded the corresponding Class<?> object
    * @return the CtClass
    */
   protected CtClass getCtClass(String name, ClassLoader classLoader)
   {
      try
      {
         return poolFactory.getPoolForLoader(classLoader).get(name);
      }
      catch (NotFoundException e)
      {
         throw raiseClassNotFound(name, e);
      }
   }

   /**
    * Gets the CtClass corresponding to {@code name} by using the bootstrap class pool.
    *
    * @param name the name of the CtClass to be retrieved
    * @return the CtClass
    */
   protected CtClass getCtClass(String name)
   {
      return getCtClass(name, null);
   }

   @Override
   @SuppressWarnings("unchecked")
   protected void generate(Class clazz, TypeInfo result)
   {
      // Everything is done lazily
   }

   public TypeInfo getTypeInfo(Class<?> clazz)
   {
      if (clazz == null)
         throw new IllegalArgumentException("Null class");

      return get(clazz);
   }


   /**
    * Gets the type info for a class
    *
    * @param name the type
    * @return the type info
    */
   public TypeInfo getTypeInfo(String name, ClassLoader cl) throws ClassNotFoundException
   {
      if (name == null)
         throw new IllegalArgumentException("Null class name");
      if (cl == null)
         cl = SecurityActions.getContextClassLoader();

      return get(name, cl);
   }

   /**
    * Gets the type info for a reflect type
    *
    * @param type the type
    * @return the type info
    */
   public TypeInfo getTypeInfo(Type type)
   {
      if (type instanceof Class)
         return getTypeInfo((Class<?>) type);
      else if (type instanceof ParameterizedType)
         return getParameterizedType((ParameterizedType)type);
      else if (type instanceof WildcardType)
         return getWildcardType((WildcardType)type);
      else if (type instanceof GenericArrayType)
         return getGenericArrayType((GenericArrayType)type);
      else if (type instanceof TypeVariable)
         return getTypeVariable((TypeVariable<?>) type);
      else
         throw new UnsupportedOperationException("Unknown type: " + type + " class=" + type.getClass());
   }

   /**
    * Get the information for a class
    *
    * @param name the name
    * @param cl the classloader
    * @return the info
    * @throws ClassNotFoundException when the class cannot be found
    */
   @Override
   public TypeInfo get(String name, ClassLoader cl) throws ClassNotFoundException
   {
      return get(name, cl, null);
   }

   /**
    * Get the information for a class
    *
    *
    * @param clazz the class
    * @return the info
    */
   @SuppressWarnings("unchecked")
   @Override
   public TypeInfo get(Class clazz)
   {
      try
      {
         ClassLoader cl = SecurityActions.getClassLoader(clazz);
         if(cl == null)
            cl = SecurityActions.getContextClassLoader();
         return get(clazz.getName(), cl, clazz);
      }
      catch (ClassNotFoundException e)
      {
         throw new RuntimeException("Class not found: "+e.getMessage());
      }
   }

   /**
    * Get the information for a class
    *
    * @param name the name
    * @param cl the classloader
    * @param clazz the class
    * @return the info
    * @throws ClassNotFoundException when the class cannot be found
    */
   protected TypeInfo get(String name, ClassLoader cl, Class<?> clazz) throws ClassNotFoundException
   {
      if (name == null)
         throw new IllegalArgumentException("Null name");
      if (cl == null)
         throw new IllegalArgumentException("Null classloader");

      return get(null, clazz, name, cl, true);
   }

   /**
    * Get the information for a class
    *
    * @param ctClass the class
    * @return the info
    */
   protected TypeInfo get(CtClass ctClass)
   {
      if (ctClass == null)
         throw new IllegalArgumentException("Null class");

      try
      {
         return get(ctClass, null, ctClass.getName(), ctClass.getClassPool().getClassLoader(), false);
      }
      catch(Exception e)
      {
         throw new RuntimeException(e);
      }
   }

   /**
    * Get the information for a class
    *
    * @param ctClass the class, may be null
    * @param clazz the class, may be null
    * @param name the name of the class
    * @param cl the class loader
    * @return the info
    * @throws IllegalArgumentException if either name or cl are null
    */
   protected TypeInfo get(CtClass ctClass, Class<?> clazz, String name, ClassLoader cl, boolean delegateToReflectIfNotFound) throws ClassNotFoundException
   {
      if (name == null)
         throw new IllegalArgumentException("Null name");
      if (cl == null)
         throw new IllegalArgumentException("Null class loader");

      TypeInfo primitive = PrimitiveInfo.valueOf(name);
      if (primitive != null)
         return primitive;

      NumberInfo number = NumberInfo.valueOf(name);
      if (number != null)
      {
         synchronized (number)
         {
            if (number.getPhase() != NumberInfo.Phase.INITIALIZING)
            {
               if (number.getPhase() != NumberInfo.Phase.COMPLETE)
               {
                  number.initializing();
                  Class<?> useClass = clazz;
                  try
                  {
                     if (useClass == null)
                        useClass = cl.loadClass(name);
                  }
                  catch (ClassNotFoundException e)
                  {
                     throw new RuntimeException(e);
                  }
                  number.setDelegate(get(useClass));
               }
               return number;
            }
         }
      }

      //Handle array names
      if (name.charAt(0) == '[')
         name = Descriptor.toClassName(name);

      Map<String, WeakReference<TypeInfo>> classLoaderCache = getClassLoaderCache(cl);
      TypeInfo result = getFromCache(name, classLoaderCache);
      if (result != null)
         return result;

      boolean changedCache = false;
      try
      {
         if (ctClass == null)
         {
            ctClass = poolFactory.getPoolForLoader(cl).get(name);
            ClassLoader realLoader = ctClass.getClassPool().getClassLoader();
            if (realLoader != null && realLoader != cl)
            {
               //Make sure that we check the cache of the classloader actually containing the class
               changedCache = true;
               classLoaderCache = getClassLoaderCache(realLoader);
               result = getFromCache(name, classLoaderCache);
               if (result != null)
                  return result;
            }
         }
         result = instantiate(ctClass, clazz);
      }
      catch(NotFoundException e)
      {
         if (delegateToReflectIfNotFound)
            result = delegateToIntrospectionImplementation(cl, name);
         else
            throw new RuntimeException(e);
      }

      if (!changedCache)
      {
         //Make sure that we cache against the classloader actually containing the class
         ClassLoader realLoader = getClassLoader(result);
         if (realLoader != cl)
            classLoaderCache = getClassLoaderCache(realLoader);
      }

      WeakReference<TypeInfo>weak = new WeakReference<TypeInfo>(result);
      classLoaderCache.put(name, weak);

//      we just ignore generate(..) since it doesnt do anything atm
//      generate(clazz, result);

      return result;
   }

   private ClassLoader getClassLoader(final TypeInfo info)
   {
      if (System.getSecurityManager() == null)
         return info.getClassLoader();

      if (info instanceof JavassistTypeInfo)
         return ((JavassistTypeInfo)info).getClassLoaderInternal();

      return AccessController.doPrivileged(new PrivilegedAction<ClassLoader>()
      {
         public ClassLoader run()
         {
            return info.getClassLoader();
         }
      });
   }

   /**
    * Proxies, whether
    * <ul>
    * <li>JDK dynamic proxies</li>
    * <li>javassist ProxyFactory proxies - created by calling ClassLoader.defineClass()</li>
    * </ul>
    * are not visible to the javassist classpools, and neither will classes generated by cglib or other
    * frameworks, so try to load up the class from the reflect implementation.
    *
    * @param cl the classloader
    * @param name the name of the class
    * @return the info
    * @throws ClassNotFoundException when the class cannot be found
    */
   private TypeInfo delegateToIntrospectionImplementation(ClassLoader cl, String name) throws ClassNotFoundException
   {
//      System.out.println("======> " + name + " " + cl);
//
//      //Extra code just for debugging
//      ClassPool pool = poolFactory.getPoolForLoader(cl);
//      try
//      {
//         CtClass ct = pool.get(name);
//      }
//      catch(Exception alreadyHandled)
//      {
//         System.out.println("---> Not found in " + pool);
//      }
//
//      Class<?> clazz = cl.loadClass(name);
//      System.out.println("---> Loaded real class from " + clazz.getClassLoader());
//
//      try
//      {
//         CtClass ct = pool.get(name);
//      }
//      catch(Exception alreadyHandled)
//      {
//         System.out.println("---> Not found again in " + pool);
//      }
//
//      pool = poolFactory.getPoolForLoader(clazz.getClassLoader());
//      try
//      {
//         CtClass ct = pool.get(name);
//         System.out.println("---> Found in actual pool " + pool);
//      }
//      catch(Exception alreadyHandled)
//      {
//         System.out.println("---> Not found in actual pool " + pool);
//      }
//

      //Extra code - END

      try
      {
         IntrospectionTypeInfoFactory factory = new IntrospectionTypeInfoFactory();
         return factory.getTypeInfo(name, cl);
      }
      catch (ClassNotFoundException e)
      {
         throw e;
      }
   }

   /**
    * Reads the type info from the cache
    *
    * @param name the class name
    * @param classLoaderCache The cached type infos for the loader
    */
   private TypeInfo getFromCache(String name, Map<String, WeakReference<TypeInfo>> classLoaderCache)
   {
      WeakReference<TypeInfo> weak = classLoaderCache.get(name);
      if (weak != null)
      {
         TypeInfo result = weak.get();
         if (result != null)
         {
            CtClass clazz = getCtClass(result, false);
            if (clazz != null && clazz.isModified())
            {
               //Only do this check if it is a Javassist implementation. delegateToIntrospectionImplementation()
               //loads up Reflect implementations in some cases
               if (!compare(clazz, (ClassInfo)result))
                  classLoaderCache.remove(name);
            }
            return result;
         }
      }
      return null;
   }

   public AnnotationValue[] getAnnotations(Object obj)
   {
      try
      {
         Object[] annotations;
         if (obj instanceof CtMember)
         {
            annotations = ((CtMember)obj).getAvailableAnnotations();
         }
         else if (obj instanceof CtClass)
         {
            annotations = ((CtClass)obj).getAvailableAnnotations();
         }
         else
         {
            throw new RuntimeException("Attempt was made to read annotations from unsupported type " + obj.getClass().getName() + ": " + obj);
         }

         if (annotations.length == 0)
         {
            return NO_ANNOTATIONS;
         }

         AnnotationValue[] annotationValues = new AnnotationValueImpl[annotations.length];
         for (int i = 0 ; i < annotations.length ; i++)
         {
            Class<?> clazz = ((Annotation)annotations[i]).annotationType();

            AnnotationInfo info = (AnnotationInfo)getTypeInfo(clazz);
            annotationValues[i] = AnnotationValueFactory.createAnnotationValue(this, this, info, annotations[i]);
         }
         return annotationValues;
      }
      catch (Throwable t)
      {
         throw new RuntimeException("Could not read annotations for " + (obj instanceof CtClass ? ((CtClass)obj).getName() : obj.toString()), t);
      }
   }

   public AnnotationValue createAnnotationValue(AnnotationInfo info, Object ann)
   {
      return AnnotationValueFactory.createAnnotationValue(this, this, info, ann);
   }


   private boolean compare(CtClass clazz, ClassInfo info)
   {
      if(clazz.getDeclaredMethods().length == info.getDeclaredMethods().length &&
            clazz.getDeclaredConstructors().length == info.getDeclaredConstructors().length &&
            clazz.getDeclaredFields().length == info.getDeclaredFields().length)
         return true;
      else
         return false;
   }

   public MutableClassInfo getMutable(String name, ClassLoader cl)
   {
      CtClass clazz;
      try
      {
         clazz = poolFactory.getPoolForLoader(cl).get(name);
         return new JavassistTypeInfo(this, clazz, null);
      }
      catch (NotFoundException e)
      {
         throw new org.jboss.reflect.spi.NotFoundException(e.toString());
      }
   }

   public MutableClassInfo createNewMutableClass(String name)
   {
      CtClass clazz = poolFactory.getPoolForLoader(null).makeClass(name);
      return new JavassistTypeInfo(this, clazz, null);
   }

   public MutableClassInfo createNewMutableClass(String name, ClassInfo superClass)
   {
      CtClass clazz = poolFactory.getPoolForLoader(superClass.getClassLoader()).makeClass(name, JavassistUtil.toCtClass(superClass));
      return new JavassistTypeInfo(this, clazz, null);
   }

   public MutableClassInfo createNewMutableInterface(String name)
   {
      CtClass clazz = poolFactory.getPoolForLoader(null).makeInterface(name);
      return new JavassistTypeInfo(this, clazz, null);
   }

   public MutableClassInfo createNewMutableInterface(String name, ClassInfo superClass)
   {
      CtClass clazz = poolFactory.getPoolForLoader(superClass.getClassLoader()).makeInterface(name, JavassistUtil.toCtClass(superClass));
      return new JavassistTypeInfo(this, clazz, null);
   }

   /**
    * Creates a type info from a reflect generic paramemeterized type
    *
    * @param type the parameterized type
    * @return the type info
    */
   protected TypeInfo getParameterizedType(ParameterizedType type)
   {
      //Create the parameterized type info
      Class<?> rawType = (Class<?>) type.getRawType();
      ClassInfo raw = (ClassInfo)getTypeInfo(rawType);
      Type[] types = type.getActualTypeArguments();

      return new JavassistParameterizedClassInfo(this, raw, types);
   }

   /**
    * Gets the type info for a javassist type
    *
    * @param loader the class loader
    * @param type the class type
    * @param spy used to determine actual types of type variables
    * @return the type info
    */
   protected TypeInfo getTypeInfo(ClassLoader loader, SignatureAttribute.Type type, JavassistTypeVariableSpy spy)
   {
      if (type instanceof ClassType)
         return getTypeInfo(loader, (ClassType)type, spy);
      else if (type instanceof ArrayType)
         return getGenericArrayType(loader, (ArrayType)type, spy);
      else if (type instanceof BaseType)
      {
         String s = String.valueOf(((BaseType)type).getDescriptor());
         Class<?> clazz = PrimitiveInfo.getPrimativeArrayComponentType(s);
         return PrimitiveInfo.valueOf(clazz.getName());
      }
      else if (type instanceof javassist.bytecode.SignatureAttribute.TypeVariable)
      {
         TypeInfo typeInfo = getTypeInfo(loader, spy.getTypeBound((javassist.bytecode.SignatureAttribute.TypeVariable)type), spy);
         if (typeInfo instanceof TypeVariableAware)
            ((TypeVariableAware)typeInfo).setTypeVariable(((javassist.bytecode.SignatureAttribute.TypeVariable)type).getName());
         return typeInfo;
      }

      //Should not happen
      throw new IllegalArgumentException("Bad type " + type + " - " + type.getClass().getName());
   }

   /**
    * Gets the type info for a javassist generic class type
    *
    * @param loader the class loader of the raw class
    * @param type the class type
    * @param spy used to determine actual types of type variables
    * @return the type info
    */
   protected TypeInfo getTypeInfo(ClassLoader loader, ClassType type, JavassistTypeVariableSpy spy)
   {
      //If it is not a parameterized type, just return the type
      boolean isParameterized = type.getTypeArguments() != null && type.getTypeArguments().length > 0;
      if (!isParameterized)
      {
         try
         {
            return get(JavassistHelper.getClassNameForGenericType(type), loader);
         }
         catch (ClassNotFoundException e)
         {
            throw new IllegalStateException(e);
         }
      }

      //Look in the cache first
      String genericName = JavassistHelper.getGenericName(type, spy);
      Map<String, WeakReference<TypeInfo>> cache = getClassLoaderCache(loader);
      TypeInfo info = getFromCache(genericName, cache);
      if (info != null)
      {
         if (info instanceof ClassInfo == false)
            throw new IllegalStateException("Not a ClassInfo " + info);
         return info;
      }

      //Create the parameterized type info
      try
      {
         ClassInfo delegate = (ClassInfo)get(JavassistHelper.getClassNameForGenericType(type), loader);
         if (!isParameterized)
            return delegate;
         info = new JavassistParameterizedClassInfo(this, delegate, loader, type.getTypeArguments(), spy);
      }
      catch (ClassNotFoundException e)
      {
         throw new IllegalStateException(e);
      }

      //Cache the parameterized type info
      cache.put(genericName, new WeakReference<TypeInfo>(info));

      return info;
   }

   /**
    * Gets the type info for a javassist type argument
    *
    * @param arg the type argument
    * @param loader the class loader
    * @param spy used to determine actual types of type variables
    * @return the type info
    */
   protected TypeInfo createTypeInfoForTypeArgument(TypeArgument arg, ClassLoader loader, JavassistTypeVariableSpy spy)
   {
      ObjectType type = arg.getType();
      if (type == null)
      {
         try
         {
            return get(Object.class.getName(), loader, null);
         }
         catch(ClassNotFoundException e)
         {
            throw new RuntimeException(e);
         }
      }
      else if (type instanceof ClassType)
      {
         return getTypeInfo(loader, (ClassType) type, spy);
      }
      else if (type instanceof javassist.bytecode.SignatureAttribute.TypeVariable)
      {
         return getTypeInfo(loader, spy.getTypeBound((javassist.bytecode.SignatureAttribute.TypeVariable)type), spy);
      }
      throw new IllegalStateException("Unhandled type " + type);
   }

   /**
    * Gets the type info for a reflect wildcard type
    *
    * @param type the wildcard type
    * @return the type info
    */
   protected TypeInfo getWildcardType(WildcardType type)
   {
      //Create the wildcard type info
      Type bound = type.getLowerBounds().length > 0 ? type.getLowerBounds()[0] : type.getUpperBounds()[0];
      return getTypeInfo(bound);
   }

   /**
    * Gets the classloader cache for the passed in classloader.
    *
    * @param cl the classloader. If null is used, the Thread context classloader will be used instead
    * @return the cache of type infos by name
    */
   @Override
   protected Map<String, WeakReference<TypeInfo>> getClassLoaderCache(ClassLoader cl)
   {
      if (cl == null)
         cl = SecurityActions.getContextClassLoader();
      return super.getClassLoaderCache(cl);
   }

   /**
    * Get the type info for a reflect generic array type
    *
    * @param type the array type
    * @return the info
    */
   protected TypeInfo getGenericArrayType(GenericArrayType type)
   {
      //Look in the cache first
      String genericName = GenericsUtil.getGenericName(type);
      ClassLoader cl = GenericsUtil.findClassLoader(type);
      Map<String, WeakReference<TypeInfo>> cache = getClassLoaderCache(cl);
      TypeInfo info = getFromCache(genericName, cache);
      if (info != null)
         return info;

      //Create the wildcard type info
      Type compType = type.getGenericComponentType();
      TypeInfo componentType = getTypeInfo(compType);

      String arrayName = getCtClass(componentType, true).getName();
      CtClass clazz = getCtClass(arrayName + "[]", cl);
      info = new JavassistArrayInfoImpl(this, clazz, null, componentType);

      //Cache the wildcard type info
      cache.put(genericName, new WeakReference<TypeInfo>(info));

      return info;
   }

   /**
    * Get the type info for a javassist array type
    *
    * @param cl the classloader
    * @param type the array type
    * @param spy used to determine actual types of type variables
    * @return the type info
    */
   protected ArrayInfo getGenericArrayType(ClassLoader cl, ArrayType type, JavassistTypeVariableSpy spy)
   {
      ModifiableArrayType wrapped = new ModifiableArrayType(type);
      return getGenericArrayType(cl, wrapped, spy);
   }

   /**
    * Recursively get the type info for a javassist array type
    *
    * @param cl the classloader
    * @param type the array type
    * @param spy used to determine actual types of type variables
    * @return the type info
    */
   protected ArrayInfo getGenericArrayType(ClassLoader cl, ModifiableArrayType type, JavassistTypeVariableSpy spy)
   {
      //Look in the cache first
      String genericName = JavassistHelper.getGenericName(type, spy);
      Map<String, WeakReference<TypeInfo>> cache = getClassLoaderCache(cl);
      ArrayInfo info = (ArrayInfo)getFromCache(genericName, cache);
      if (info != null)
         return info;

      type.decrement();
      TypeInfo componentType = type.getDimension() > 0 ?
         getGenericArrayType(cl, type, spy) : getTypeInfo(cl, type.getComponentType(), spy);

      String arrayName = getCtClass(componentType, true).getName();
      CtClass clazz = getCtClass(arrayName + "[]", cl);
      info = new JavassistArrayInfoImpl(this, clazz, null, componentType);


      //Cache the wildcard type info
      cache.put(genericName, new WeakReference<TypeInfo>(info));

      return info;
   }

   /**
    * Gets the CtClass from a TypeInfo if it is a javassist one
    *
    * @param typeInfo the type info
    * @param error if true and the type can not be determined, throw an error
    * @return null if the type could not be determined and <code>error</code> is false
    * @throws IllegalArgumentException if <code>error</code> is true and the type can not be determined
    */
   private CtClass getCtClass(TypeInfo typeInfo, boolean error)
   {
      if (typeInfo instanceof PrimitiveInfo)
         return getPrimitiveCtClass(typeInfo.getName());
      if (typeInfo instanceof JavassistTypeInfo)
         return ((JavassistTypeInfo)typeInfo).getCtClass();
      if (typeInfo instanceof DelegateClassInfo)
         return getCtClass(((DelegateClassInfo) typeInfo).getDelegate(), error);

      if (error)
         throw new IllegalArgumentException(typeInfo + " is not a JavassistType info, a PrimitiveTypeInfo or a JavassistParameterizedType");

      return null;
   }

   private CtClass getPrimitiveCtClass(String name)
   {
      try
      {
         return ClassPool.getDefault().get(name);
      }
      catch(NotFoundException e)
      {
         throw new IllegalStateException("Could not find primitive called " + name, e);
      }
   }

   /**
    * Get the type info for a reflect TypeVariable
    *
    * @param type the type variable
    * @return the type info
    *
    */
   protected TypeInfo getTypeVariable(TypeVariable<?> type)
   {
      return getTypeInfo(type.getBounds()[0]);
   }

   /**
    * Wrapper around {@link ArrayType} to make it easier to
    * create the name used to look up the array type infos
    * in the cache
    */
   protected static class ModifiableArrayType extends ArrayType
   {
      int dims;

      ModifiableArrayType(ArrayType original)
      {
         super(original.getDimension(), original.getComponentType());
         dims = original.getDimension();
      }

      @Override
      public int getDimension()
      {
         return dims;
      }

      void decrement()
      {
         dims--;
      }
   }
}
