/*
 * JBoss, Home of Professional Open Source.
 *
 * Copyright 2018 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.logmanager.ext.util;

import java.util.Collections;
import java.util.EnumMap;
import java.util.LinkedHashMap;
import java.util.Map;

import org.jboss.logmanager.ext.formatters.StructuredFormatter.Key;
import org.junit.Assert;
import org.junit.Test;

/**
 * @author <a href="mailto:jperkins@redhat.com">James R. Perkins</a>
 */
public class PropertyValuesTests {

    @Test
    public void testStringToMap() {
        Map<String, String> map = MapBuilder.<String, String>create()
                .add("key1", "value1")
                .add("key2", "value2")
                .add("key3", "value3")
                .build();
        Map<String, String> parsedMap = PropertyValues.stringToMap("key1=value1,key2=value2,key3=value3");
        compareMaps(map, parsedMap);

        map = MapBuilder.<String, String>create()
                .add("key=1", "value1")
                .add("key=2", "value,2")
                .add("key3", "value,3")
                .build();
        parsedMap = PropertyValues.stringToMap("key\\=1=value1,key\\=2=value\\,2,key3=value\\,3");
        compareMaps(map, parsedMap);

        map = MapBuilder.<String, String>create()
                .add("key=", "value,")
                .add("key2", "value2")
                .add("key\\", "value\\")
                .add("this", "some=thing\\thing=some")
                .build();
        parsedMap = PropertyValues.stringToMap("key\\==value\\,,key2=value2,key\\\\=value\\\\,this=some=thing\\\\thing=some");
        compareMaps(map, parsedMap);

        map = MapBuilder.<String, String>create()
                .add("key1", "value1")
                .add("key2", null)
                .add("key3", "value3")
                .add("key4", null)
                .build();
        parsedMap = PropertyValues.stringToMap("key1=value1,key2=,key3=value3,key4");
        compareMaps(map, parsedMap);

        map = MapBuilder.<String, String>create()
                .add("company", "Red Hat, Inc.")
                .add("product", "JBoss")
                .add("name", "First \"nick\" Last")
                .build();
        parsedMap = PropertyValues.stringToMap("company=Red Hat\\, Inc.,product=JBoss,name=First \"nick\" Last");
        compareMaps(map, parsedMap);

        Assert.assertTrue("Map is not empty", PropertyValues.stringToMap(null).isEmpty());
        Assert.assertTrue("Map is not empty", PropertyValues.stringToMap("").isEmpty());
    }

    @Test
    public void testStringToMapValueExpressions() {
        Map<String, String> map = MapBuilder.<String, String>create()
                .add("key1", "${org.jboss.logmanager.test.sysprop1}")
                .add("key2=", "${org.jboss.logmanager.test.sysprop2}")
                .build();
        Map<String, String> parsedMap = PropertyValues.stringToMap("key1=${org.jboss.logmanager.test.sysprop1},key2\\==${org.jboss.logmanager.test.sysprop2}");
        compareMaps(map, parsedMap);
    }

    @Test
    public void testStringToEnumMap() throws Exception {
        Map<Key, String> map = MapBuilder.<Key, String>create()
                .add(Key.EXCEPTION_CAUSED_BY, "cause")
                .add(Key.MESSAGE, "msg")
                .build();
        EnumMap<Key, String> parsedMap = PropertyValues.stringToEnumMap(Key.class, "EXCEPTION_CAUSED_BY=cause,MESSAGE=msg,HOST_NAME=hostname");
        compareMaps(map, parsedMap);

        parsedMap = PropertyValues.stringToEnumMap(Key.class, "exception-caused-by=cause,message=msg,host-name=hostname");
        compareMaps(map, parsedMap);
    }

    @Test
    public void testMapToString() throws Exception {
        Map<String, String> map = MapBuilder.<String, String>create()
                .add("key1", "value1")
                .add("key2", "value2")
                .add("key3", "value3")
                .build();
        Assert.assertEquals("key1=value1,key2=value2,key3=value3", PropertyValues.mapToString(map));

        map = MapBuilder.<String, String>create()
                .add("key=1", "value1")
                .add("key=2", "value,2")
                .add("key3", "value,3")
                .build();
        Assert.assertEquals("key\\=1=value1,key\\=2=value\\,2,key3=value\\,3", PropertyValues.mapToString(map));

        map = MapBuilder.<String, String>create()
                .add("key=", "value,")
                .add("key2", "value2")
                .add("key\\", "value\\")
                .add("this", "some=thing\\thing=some")
                .build();
        Assert.assertEquals("key\\==value\\,,key2=value2,key\\\\=value\\\\,this=some=thing\\\\thing=some", PropertyValues.mapToString(map));

        map = MapBuilder.<String, String>create()
                .add("key1", "value1")
                .add("key2", null)
                .add("key3", "value3")
                .add("key4", null)
                .build();
        Assert.assertEquals("key1=value1,key2=,key3=value3,key4=", PropertyValues.mapToString(map));

        map = MapBuilder.<String, String>create()
                .add("company", "Red Hat, Inc.")
                .add("product", "JBoss")
                .add("name", "First \"nick\" Last")
                .build();
        Assert.assertEquals("company=Red Hat\\, Inc.,product=JBoss,name=First \"nick\" Last", PropertyValues.mapToString(map));

        Assert.assertTrue("Expected an empty map", PropertyValues.stringToMap(null).isEmpty());
        Assert.assertTrue("Expected an empty map", PropertyValues.stringToMap("").isEmpty());
    }

    /**
     * Compares the two maps have the same keys and same values in any order.
     *
     * @param m1  the first map used to compare the keys and values
     * @param m2  the second map used to compare the keys and values
     * @param <K> the key type
     * @param <V> the value type
     */
    @SuppressWarnings("WeakerAccess")
    private static <K, V> void compareMaps(final Map<K, V> m1, final Map<K, V> m2) {
        String failureMessage = String.format("Keys did not match%n%s%n%s%n", m1.keySet(), m2.keySet());
        Assert.assertTrue(failureMessage, m1.keySet().containsAll(m2.keySet()));
        Assert.assertTrue(failureMessage, m2.keySet().containsAll(m1.keySet()));

        // At this point we know that all the keys match
        for (K key : m1.keySet()) {
            final V value1 = m1.get(key);
            final V value2 = m2.get(key);
            Assert.assertEquals(
                    String.format("Value %s from the first map does not match value %s from the second map with key %s.", value1, value2, key),
                    value1, value2);
        }
    }

    /**
     * A helper to easily build maps. The resulting map is immutable and the order is predictable with the
     * {@link #add(Object, Object)} order.
     *
     * @author <a href="mailto:jperkins@redhat.com">James R. Perkins</a>
     */
    private static class MapBuilder<K, V> {
        private final Map<K, V> result;

        private MapBuilder(final Map<K, V> result) {
            this.result = result;
        }

        public static <K, V> MapBuilder<K, V> create() {
            return new MapBuilder<>(new LinkedHashMap<K, V>());
        }

        public MapBuilder<K, V> add(final K key, final V value) {
            result.put(key, value);
            return this;
        }

        public Map<K, V> build() {
            return Collections.unmodifiableMap(result);
        }
    }
}
