/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.bean.impl.manager;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.interceptor.InvocationContext;

import org.jboss.injection.manager.spi.InjectionManager;
import org.jboss.interceptor.proxy.DefaultInvocationContextFactory;
import org.jboss.interceptor.proxy.InterceptorInvocation;
import org.jboss.interceptor.proxy.SimpleInterceptionChain;
import org.jboss.interceptor.spi.context.InterceptionChain;
import org.jboss.interceptor.spi.metadata.InterceptorMetadata;
import org.jboss.interceptor.spi.model.InterceptionType;
import org.jboss.managed.bean.impl.ManagedBeanInstanceImpl;
import org.jboss.managed.bean.metadata.ManagedBeanMetaData;
import org.jboss.managed.bean.spi.ManagedBeanInstance;
import org.jboss.managed.bean.spi.ManagedBeanManager;

/**
 * Manages a managed bean.
 * 
 * <p>
 *  {@link ManagedBeanManagerImpl} is responsible for managing the runtime of a managed bean. This includes 
 *  (although not limited to):
 *  
 *   <ul>
 *      <li>Instance creation of the managed bean either directly or indirectly</li>
 *      <li>Invoking appropriate lifecycle callbacks on the managed bean</li>
 *      <li>Passing on the managed bean instance through a chain on injectors</li>
 *      <li>Passing on the invocations on a managed bean instance, through a chain of interceptors</li>  
 *   </ul>
 * Note that the {@link ManagedBeanManagerImpl} may utilize services from other SPIs for any/all of the above responsibilities.    
 * </p>
 *
 * FIXME: After an initial working version, move this {@link ManagedBeanManagerImpl} as an SPI
 * after deciding the contracts 
 *  
 * @author Jaikiran Pai
 * @version $Revision: $
 */
// Implementation note: Do NOT include any MC (or any other kernel) specific implicit/explicit semantics (like start() stop() etc...) in
// this manager 
public class ManagedBeanManagerImpl<T> implements ManagedBeanManager<T>
{

   /**
    * Managed bean metadata
    */
   protected ManagedBeanMetaData mbMetaData;

   /**
    * The managed bean class
    */
   protected Class<T> managedBeanClass;
   
   private InjectionManager injectionManager;
   
   
   /**
    * Constructs a {@link ManagedBeanManagerImpl} for the passed managed bean class and metadata.
    * 
    * @param beanClass The managed bean class
    * @param beanMetaData The metadata of the managed bean
    * 
    */
   public ManagedBeanManagerImpl(Class<T> beanClass, ManagedBeanMetaData beanMetaData, InjectionManager injectionManager)
   {
      if (beanClass == null || beanMetaData == null || injectionManager == null)
      {
         throw new IllegalArgumentException("Neither of managed bean class " + beanClass + " ,managed bean metadata "
               + beanMetaData + " nor injection manager " + injectionManager + " should be null");
      }
      this.managedBeanClass = beanClass;
      this.mbMetaData = beanMetaData;
      this.injectionManager = injectionManager;
   }
   
   @Override
   public ManagedBeanMetaData getMetadata()
   {
      return this.mbMetaData;
   }

   @SuppressWarnings("unchecked")
   @Override
   public Object invoke(ManagedBeanInstance<T> managedBeanInstance, Method method, Object[] args) throws Throwable
   {
      Collection<InterceptorMetadata<?>> aroundInvokeInterceptors = this.mbMetaData.getAroundInvokes(method);
      Collection<InterceptorInvocation<?>> interceptorInvocations = new ArrayList<InterceptorInvocation<?>>(aroundInvokeInterceptors.size());
      for (InterceptorMetadata<?> aroundInvokeInterceptor : aroundInvokeInterceptors)
      {
         Object instance = null;
         if (aroundInvokeInterceptor.isTargetClass())
         {
            instance = managedBeanInstance.getInstance();
         }
         else
         {
            String interceptorClassName = aroundInvokeInterceptor.getInterceptorClass().getClassName();
            instance = managedBeanInstance.getInterceptor(interceptorClassName);
         }
         // TODO: Interceptor invocation creation should be simplified in jboss-interceptors project.
         InterceptorInvocation<?> interceptorInvocation = new InterceptorInvocation(instance, aroundInvokeInterceptor, InterceptionType.AROUND_INVOKE);
         interceptorInvocations.add(interceptorInvocation);
      }
      InterceptionChain interceptorChain = new SimpleInterceptionChain(interceptorInvocations,
            InterceptionType.AROUND_INVOKE, managedBeanInstance.getInstance(), method);
      DefaultInvocationContextFactory invocationCtxFactory = new DefaultInvocationContextFactory();
      InvocationContext invocationCtx = invocationCtxFactory.newInvocationContext(interceptorChain, managedBeanInstance.getInstance(), method, args);
      // invoke
      return invocationCtx.proceed();
   }


   @Override
   public ManagedBeanInstance<T> createManagedBeanInstance() throws Exception
   {
      T instance = this.createInstance(this.managedBeanClass);
      Collection<Object> interceptorInstances = this.createInterceptors();
      // inject the MB instance
      this.injectionManager.inject(instance);
      // inject the interceptor instances
      for (Object interceptorInstance : interceptorInstances)
      {
         this.injectionManager.inject(interceptorInstance);
      }
      
      ManagedBeanInstance<T> mbInstance = new ManagedBeanInstanceImpl<T>(instance, interceptorInstances);
      // invoke post-construct
      this.handlePostConstruct(mbInstance);
      return mbInstance;
   }

   @SuppressWarnings("unchecked")
   private void handlePostConstruct(ManagedBeanInstance<T> managedBeanInstance) throws Exception
   {
      List<InterceptorMetadata<?>> postConstructs = this.mbMetaData.getPostConstructs();
      Collection<InterceptorInvocation<?>> interceptorInvocations = new ArrayList<InterceptorInvocation<?>>(postConstructs.size());
      for (InterceptorMetadata<?> interceptor : postConstructs)
      {
         Object instance = null;
         if (interceptor.isTargetClass())
         {
            instance = managedBeanInstance.getInstance();
         }
         else
         {
            String interceptorClassName = interceptor.getInterceptorClass().getClassName();
            instance = managedBeanInstance.getInterceptor(interceptorClassName);
         }
         // TODO: Interceptor invocation creation will be simplified in jboss-interceptors project.
         InterceptorInvocation<?> interceptorInvocation = new InterceptorInvocation(instance, interceptor, InterceptionType.POST_CONSTRUCT);
         interceptorInvocations.add(interceptorInvocation);
      }
      InterceptionChain interceptorChain = new SimpleInterceptionChain(interceptorInvocations,
            InterceptionType.POST_CONSTRUCT, managedBeanInstance.getInstance(), null);
      DefaultInvocationContextFactory invocationCtxFactory = new DefaultInvocationContextFactory();
      InvocationContext invocationCtx = invocationCtxFactory.newInvocationContext(interceptorChain, managedBeanInstance.getInstance(), null, null);
      // invoke post-construct
      invocationCtx.proceed();
   }
   
   private Collection<Object> createInterceptors()
   {
      Collection<InterceptorMetadata<?>> allInterceptors = this.mbMetaData.getAllInterceptors();
      Collection<Object> interceptorInstances = new ArrayList<Object>();
      for (InterceptorMetadata<?> interceptor : allInterceptors)
      {
         String interceptorClassName = interceptor.getInterceptorClass().getClassName();
         try
         {
            Class<?> interceptorClass = Class.forName(interceptorClassName, false, this.managedBeanClass
                  .getClassLoader());
            Object interceptorInstance = this.createInstance(interceptorClass);
            interceptorInstances.add(interceptorInstance);
         }
         catch (ClassNotFoundException cnfe)
         {
            throw new RuntimeException("Could not load interceptor class: " + interceptorClassName, cnfe);
         }

      }
      return interceptorInstances;
   }

   private <I> I createInstance(Class<I> klass)
   {
      try
      {
         // for now just ignore the targetIdentifier and create a new instance
         return klass.newInstance();
      }
      catch (InstantiationException ie)
      {
         // TODO: I guess a BeanInstantionException SPI would be better.
         // But let's wait for the decision on the jboss-ejb3-bean-instantiator usage
         throw new RuntimeException(ie);
      }
      catch (IllegalAccessException iae)
      {
         // TODO: I guess a BeanInstantionException SPI would be better.
         // But let's wait for the decision on the jboss-ejb3-bean-instantiator usage
         throw new RuntimeException(iae);
      }
   }

}
