/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.bean.mc.resource.provider;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.managed.bean.metadata.ManagedBeanDeploymentMetaData;
import org.jboss.managed.bean.metadata.ManagedBeanMetaData;

/**
 * Base class for resolving a managed bean resource reference in deployment units
 *
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public abstract class AbstractManagedBeanRefResourceProvider
{

   /**
    * Returns the managed bean (if any) whose managed bean class name is equal to the passed <code>mbClassName</code>
    * <p>
    *   This method processes even the nested deployment unit(s) and the parent unit(s) of the passed deployment unit. 
    * </p>
    * @param unit The deployment unit being scanned
    * @param alreadyScannedDUs The set of deployment units which have already been scanned 
    * @param mbClassName The fully qualified class name of the Managed Bean
    * @return
    */
   protected ManagedBeanMetaData resolveWithinDeploymentUnit(DeploymentUnit unit, Set<DeploymentUnit> alreadyScannedDUs, String mbClassName)
   {
      // first find in the passed DU
      ManagedBeanMetaData managedBean = this.getManagedBean(unit, mbClassName);
      // found, just return it
      if (managedBean != null)
      {
         return managedBean;
      }

      if (alreadyScannedDUs == null)
      {
         alreadyScannedDUs = new HashSet<DeploymentUnit>();
      }

      // Not resolved in the passed DU, so let's
      // check try resolving in its children DUs
      List<DeploymentUnit> children = unit.getChildren();
      if (children != null)
      {
         for (DeploymentUnit child : children)
         {
            // already searched that one
            if (alreadyScannedDUs.contains(child))
            {
               continue;
            }
            // try resolving in this child DU
            managedBean = resolveWithinDeploymentUnit(child, alreadyScannedDUs, mbClassName);
            // found in this child DU (or its nested child), return the managed bean
            if (managedBean != null)
            {
               return managedBean;
            }
            // add the child DU to the already scanned DU collection
            // so that we don't scan it again
            alreadyScannedDUs.add(child);
         }
      }

      // add this DU to the already scanned DU collection
      alreadyScannedDUs.add(unit);

      // we haven't yet resolved the managed bean, so let's
      // try resolving in our parent (and any of its children)
      DeploymentUnit parent = unit.getParent();
      if (parent != null)
      {
         return resolveWithinDeploymentUnit(parent, alreadyScannedDUs, mbClassName);
      }
      // couldn't resolve in the entire DU hierarchy, return null
      return null;
   }
   
   /**
    * Returns the managed bean (if any) whose managed bean class name is equal to the passed <code>mbClassName</code>
    * @param unit The deployment unit within which the Managed Bean is scanned for. Note that, the child of parent
    *               deployment units of the passed <code>unit</code> will <i>not</i> be looked into
    * @param mbClassName The fully qualified class name of the managed bean being searched
    * @return
    */
   private ManagedBeanMetaData getManagedBean(DeploymentUnit unit, String mbClassName)
   {
      ManagedBeanDeploymentMetaData managedBeanDeploymentMetaData = unit.getAttachment(ManagedBeanDeploymentMetaData.class);
      if (managedBeanDeploymentMetaData == null)
      {
         return null;
      }
      Collection<ManagedBeanMetaData> managedBeans = managedBeanDeploymentMetaData.getManagedBeans();
      if (managedBeans == null || managedBeans.isEmpty())
      {
         return null;
      }
      for (ManagedBeanMetaData managedBean : managedBeans)
      {
         if (mbClassName.equals(managedBean.getManagedBeanClass()))
         {
            return managedBean;
         }
      }
      return null;
   }
   
}
