/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.bean.mc.deployer;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;

import org.jboss.beans.metadata.api.annotations.Inject;
import org.jboss.beans.metadata.plugins.builder.BeanMetaDataBuilderFactory;
import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.injection.injector.EEInjector;
import org.jboss.injection.injector.metadata.EnvironmentEntryType;
import org.jboss.injection.injector.metadata.InjectionTargetType;
import org.jboss.injection.injector.metadata.JndiEnvironmentRefsGroup;
import org.jboss.injection.manager.spi.InjectionManager;
import org.jboss.injection.manager.spi.Injector;
import org.jboss.injection.mc.metadata.JndiEnvironmentImpl;
import org.jboss.logging.Logger;
import org.jboss.managed.bean.impl.manager.ManagedBeanManagerImpl;
import org.jboss.managed.bean.mc.util.ManagedBeanMCBeanUtil;
import org.jboss.managed.bean.metadata.ManagedBeanDeploymentMetaData;
import org.jboss.managed.bean.metadata.ManagedBeanMetaData;
import org.jboss.managed.bean.metadata.jbmeta.JBMetaManagedBeanMetaData;
import org.jboss.managed.bean.spi.ManagedBeanManager;
import org.jboss.reloaded.naming.deployers.javaee.JavaEEComponentInformer;
import org.jboss.switchboard.spi.Barrier;

/**
 * Responsible for picking up {@link DeploymentUnit}s which have {@link ManagedBeanDeploymentMetaData}
 * and create a {@link ManagedBeanManagerImpl} for each of the {@link ManagedBeanMetaData} contained in
 * that {@link ManagedBeanDeploymentMetaData}
 *
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public class ManagedBeanManagerDeployer extends AbstractDeployer
{
   /**
    * Logger
    */
   private static Logger logger = Logger.getLogger(ManagedBeanManagerDeployer.class);
   
   /**
    * component informer
    */
   private JavaEEComponentInformer javaeeComponentInformer;

   public ManagedBeanManagerDeployer()
   {
      this.setStage(DeploymentStages.REAL);

      this.setInput(ManagedBeanDeploymentMetaData.class);

      // switchboard
      this.addInput(Barrier.class);

      // we output MC bean metadata
      this.setOutput(BeanMetaData.class);
   }

   @Inject
   public void setJavaEEComponentInformer(JavaEEComponentInformer componentInformer)
   {
      this.javaeeComponentInformer = componentInformer;
   }

   @Override
   public void deploy(DeploymentUnit unit) throws DeploymentException
   {
      ManagedBeanDeploymentMetaData managedBeanDeployment = unit.getAttachment(ManagedBeanDeploymentMetaData.class);

      Collection<ManagedBeanMetaData> managedBeans = managedBeanDeployment.getManagedBeans();
      if (managedBeans == null || managedBeans.isEmpty())
      {
         return;
      }

      for (ManagedBeanMetaData managedBean : managedBeans)
      {
         this.deploy(unit, managedBean);
      }

   }

   private void deploy(DeploymentUnit unit, ManagedBeanMetaData managedBean) throws DeploymentException
   {
      Class<?> beanClass;
      try
      {
         beanClass = this.getBeanClass(unit, managedBean);
      }
      catch (ClassNotFoundException cnfe)
      {
         throw new DeploymentException("Could not load managed bean class " + managedBean.getManagedBeanClass()
               + " in unit " + unit, cnfe);
      }
      InjectionManager injectionManager = unit.getAttachment(InjectionManager.class);
      String managedBeanManagerMCBeanName = ManagedBeanMCBeanUtil.getManagedBeanManagerMCBeanName(this.javaeeComponentInformer, unit, managedBean);
      ManagedBeanManager<?> managedBeanManager = new ManagedBeanManagerImpl(beanClass, managedBean, injectionManager);
      BeanMetaDataBuilder builder = BeanMetaDataBuilderFactory.createBuilder(managedBeanManagerMCBeanName,
            ManagedBeanManagerImpl.class.getName());

      builder.setConstructorValue(managedBeanManager);

      // setup switchboard dependency on the managed bean manager
      Barrier switchboard = unit.getAttachment(Barrier.class);
      logger.debug("Adding switchboard dependency: " + switchboard.getId() + " for managed bean manager: "
            + managedBeanManagerMCBeanName + " in unit " + unit);
      builder.addDependency(switchboard.getId());
      
      // setup the EEInjector dependency on the managed bean manager
      Set<String> injectors = this.setupInjectors(unit, managedBeanManager, injectionManager, switchboard);
      if (injectors != null)
      {
         for (String injectorMCBeanName : injectors)
         {
            logger.debug("Adding injector dependency: " + injectorMCBeanName + " for managed bean manager: "
                  + managedBeanManagerMCBeanName + " in unit " + unit);
            builder.addDependency(injectorMCBeanName);
         }
      }

      unit.addAttachment(BeanMetaData.class + ":" + managedBeanManagerMCBeanName, builder.getBeanMetaData(),
            BeanMetaData.class);

   }

   private Class<?> getBeanClass(DeploymentUnit unit, ManagedBeanMetaData managedBean) throws ClassNotFoundException
   {
      ClassLoader cl = unit.getClassLoader();

      return Class.forName(managedBean.getManagedBeanClass(), false, cl);
   }

   private Set<String> setupInjectors(DeploymentUnit unit, ManagedBeanManager<?> managedBeanManager, InjectionManager injectionManager, Barrier switchBoard)
   {
      // Let's first create EEInjector (which pulls from ENC and pushes to EJB instance) for EJB
      // and then create a EEInjector for each of the interceptors for the bean

      // Hack!
      if (!(managedBeanManager.getMetadata() instanceof JBMetaManagedBeanMetaData))
      {
         return null;
      }
      JBMetaManagedBeanMetaData managedBeanMetaData = (JBMetaManagedBeanMetaData) managedBeanManager.getMetadata();
      if (managedBeanMetaData.getEnvironment() != null)
      {
         // convert JBMETA metadata to jboss-injection specific metadata
         JndiEnvironmentRefsGroup jndiEnvironment = new JndiEnvironmentImpl(managedBeanMetaData.getEnvironment(), unit
               .getClassLoader());
         // For optimization, we'll create an Injector only if there's atleast one InjectionTarget
         if (this.hasInjectionTargets(jndiEnvironment))
         {
            // create the injector
            EEInjector eeInjector = new EEInjector(jndiEnvironment);
            // add the injector the injection manager
            injectionManager.addInjector(eeInjector);
            // Deploy the Injector as a MC bean (so that the fully populated naming context (obtained via the SwitchBoard
            // Barrier) gets injected.
            String injectorMCBeanName = this.getInjectorMCBeanNamePrefix(unit) + ",managedbean="
                  + managedBeanMetaData.getName();
            BeanMetaData injectorBMD = this.createInjectorBMD(injectorMCBeanName, eeInjector, switchBoard);
            unit.addAttachment(BeanMetaData.class + ":" + injectorMCBeanName, injectorBMD);

            return Collections.singleton(injectorMCBeanName);
         }
      }
      return null;
   }

   /**
    * Returns true if the passed {@link JndiEnvironmentRefsGroup} has atleast one {@link EnvironmentEntryType environment entry}
    * with an {@link InjectionTargetType injection target}. Else, returns false
    *  
    * @param jndiEnv
    * @return
    */
   private boolean hasInjectionTargets(JndiEnvironmentRefsGroup jndiEnv)
   {
      Collection<EnvironmentEntryType> envEntries = jndiEnv.getEntries();
      if (envEntries == null || envEntries.isEmpty())
      {
         return false;
      }
      for (EnvironmentEntryType envEntry : envEntries)
      {
         Collection<InjectionTargetType> injectionTargets = envEntry.getInjectionTargets();
         if (injectionTargets != null && !injectionTargets.isEmpty())
         {
            return true;
         }
      }
      return false;
   }

   /**
    * Creates and returns {@link BeanMetaData} for the passed {@link EEInjector injector} and sets up
    * dependency on the passed {@link Barrier SwitchBoard barrier}.
    * 
    * @param injectorMCBeanName
    * @param injector
    * @param barrier
    * @return
    */
   private BeanMetaData createInjectorBMD(String injectorMCBeanName, EEInjector injector, Barrier barrier)
   {
      BeanMetaDataBuilder builder = BeanMetaDataBuilderFactory.createBuilder(injectorMCBeanName, injector.getClass()
            .getName());
      builder.setConstructorValue(injector);

      // add dependency on INSTALLED state of SwitchBoard Barrier
      builder.addDependency(barrier.getId());

      // return the Injector BMD
      return builder.getBeanMetaData();
   }

   /**
    * Returns the prefix for the MC bean name, for a {@link Injector injector}
    * 
    * @return
    */
   private String getInjectorMCBeanNamePrefix(DeploymentUnit unit)
   {
      StringBuilder sb = new StringBuilder("jboss-injector:");
      org.jboss.deployers.structure.spi.DeploymentUnit topLevelUnit = unit.isTopLevel() ? unit : unit.getTopLevel();
      sb.append("topLevelUnit=");
      sb.append(topLevelUnit.getSimpleName());
      sb.append(",");
      sb.append("unit=");
      sb.append(unit.getSimpleName());

      return sb.toString();
   }

}
