/*
* JBoss, Home of Professional Open Source
* Copyright 2007, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.jmx.mbeanserver;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.RuntimeOperationsException;

/**
 * JBossMBeanRegistry.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public class JBossMBeanRegistry
{
   /** The current registry entry */
   static ThreadLocal<RegistryEntry> current = new ThreadLocal<RegistryEntry>();
   
   /**
    * A map of domain name to another map containing object name canonical
    * key properties to registry entries.
    * domain -> canonicalKeyProperties -> MBeanEntry
    */
   private Map<String, Map<String, RegistryEntry>> domainMap = new ConcurrentHashMap<String, Map<String, RegistryEntry>>();

   /** The default domain for this registry */
   private String defaultDomain;

   /** The MBeanServer for which we are the registry. */
   private MBeanServer server;

   /**
    * Create a new JBossMBeanRegistry.
    * 
    * @param server the server
    */
   public JBossMBeanRegistry(MBeanServer server)
   {
      this.server = server;
      defaultDomain = server.getDefaultDomain();
   }

   /**
    * Register an mbean.<p>
    *
    * The object name passed maybe unqualified.<p>
    *
    * The map is a user definable string to value object map for holding
    * information against a registered object. This map may contain metadata
    * related to the registration, such as registration date/time, classloader
    * references, etc. <p>
    *
    * Pass org.jboss.mx.classloader in the values map to
    * set the context classloader<p>
    *
    * Other values are user definable and can be retrieved using the
    * getValue(ObjectName,String) method.
    *
    * @param object     the mbean to register.
    * @param name       the object name to assign to the mbean.
    * @param valueMap   a map of other information to include in the
    *                   registry
    *
    * @return an object instance for the registered mbean
    *
    * @exception InstanceAlreadyExistsException when the object name
    *            is already registered.
    * @exception MBeanRegistrationException when an exception is raised
    *            during preRegister for the mbean.
    * @exception NotCompliantMBeanException when the passed object is
    *            a valid mbean.
    */
   public ObjectInstance registerMBean(Object object, ObjectName name, Map<String, Object> valueMap)
      throws InstanceAlreadyExistsException, MBeanRegistrationException, NotCompliantMBeanException
   {
      RegistryEntry entry = new RegistryEntry(qualifyName(name), valueMap);
      RegistryEntry previous = current.get();
      current.set(entry);
      try
      {
         return server.registerMBean(object, entry.getObjectName());
      }
      finally
      {
         current.set(previous);
      }
   }


   /**
    * Unregister an mbean. <p>
    *
    * This method is invoked by the MBeanServer for
    * unregisterMBean().<p>
    *
    * The object name passed maybe unqualified.<p>
    *
    * MBeans in JMImplementation cannot be unregistered
    *
    * @param name the mbean to unregister.
    *
    * @exception InstanceNotFoundException when the object name is
    *            not registered.
    * @exception MBeanRegistrationException when an exception is raised
    *            during preDeregister for the mbean.
    */
   public void unregisterMBean(ObjectName name)
      throws InstanceNotFoundException, MBeanRegistrationException
   {
      RegistryEntry entry = new RegistryEntry(qualifyName(name), null);
      RegistryEntry previous = current.get();
      current.set(entry);
      try
      {
         server.unregisterMBean(name);
      }
      finally
      {
         current.set(previous);
      }
   }

   /**
    * Retrieve the registration for an object name.<p>
    *
    * This method is invoked by the MBeanServer for
    * methods passing an ObjectName that are not covered in other methods.<p>
    *
    * The object name passed maybe unqualified.
    *
    * @param name the object name to retrieve
    *
    * @return the mbean's registration
    *
    * @exception InstanceNotFoundException when the object name is not
    *            registered.
    */
   public RegistryEntry get(ObjectName name)
      throws InstanceNotFoundException
   {
      return get(name, true);
   }

   /**
    * Retrieve the registration for an object name.<p>
    *
    * This method is invoked by the MBeanServer for
    * methods passing an ObjectName that are not covered in other methods.<p>
    *
    * The object name passed maybe unqualified.
    *
    * @param name the object name to retrieve
    * @param errorIfNotFound whether to throw an error if not found
    * @return the mbean's registration
    *
    * @exception InstanceNotFoundException when the object name is not
    *            registered.
    */
   public RegistryEntry get(ObjectName name, boolean errorIfNotFound)
      throws InstanceNotFoundException
   {
      if (name == null)
         throw new RuntimeOperationsException(new IllegalArgumentException("null object name"));

      // Determine the domain and retrieve its entries
      String domain = name.getDomain();

      if (domain.length() == 0)
         domain = defaultDomain;

      String props = name.getCanonicalKeyPropertyListString();
      Map<String, RegistryEntry> mbeanMap = getMBeanMap(domain, false);

      // Retrieve the registry entry
      RegistryEntry entry = null;
      if ((null == mbeanMap || null == (entry = mbeanMap.get(props))) && errorIfNotFound)
         throw new InstanceNotFoundException(name + " is not registered.");

      // We are done
      return entry;
   }
   
   /**
    * Retrieve the default domain for this registry.<p>
    *
    * @return the default domain
    */
   public String getDefaultDomain()
   {
      return server.getDefaultDomain();
   }

   /**
    * Retrieve the domains for this registry.<p>
    *
    * @return the domains
    */
   public String[] getDomains()
   {
      return server.getDomains();
   }

   public ObjectInstance getObjectInstance(ObjectName name)
      throws InstanceNotFoundException
   {
      return server.getObjectInstance(name);
   }

   /**
    * Retrieve the object instance for an object name.
    *
    * @param name the object name of the mbean
    * @param key the key to retrieve
    * @return the object instance
    * @exception InstanceNotFoundException when the object name is not
    *            registered
    */
   public Object getValue(ObjectName name, String key)
      throws InstanceNotFoundException
   {
      return get(name).getValue(key);
   }

   /**
    * Test whether an object name is registered. <p>
    *
    * This method is invoked by the MBeanServer for
    * isRegistered().<p>
    *
    * The object name passed maybe unqualified.
    *
    * @param name the object name
    * @return true when the object name is registered, false otherwise
    */
   public boolean contains(ObjectName name)
   {
      return server.isRegistered(name);
   }

   /**
    * Retrieve the number of mbeans registered.<p>
    *
    * This method is invoked by the MBeanServer for
    * getMBeanCount().
    *
    * @return the number of mbeans registered.
    */
   public int getSize()
   {
      return server.getMBeanCount();
   }

   /**
    * Adds an MBean entry<p>
    *
    * WARNING: The object name should be fully qualified.
    *
    * @param entry the MBean entry to add
    * @exception InstanceAlreadyExistsException when the MBean's object name
    *            is already registered
    */
   protected synchronized void add(RegistryEntry entry)
      throws InstanceAlreadyExistsException
   {
      // Determine the MBean's name and properties
      ObjectName name = entry.getObjectName();
      String domain = name.getDomain();
      String props = name.getCanonicalKeyPropertyListString();

      // Create a properties -> entry map if we don't have one
      Map<String, RegistryEntry> mbeanMap = getMBeanMap(domain, true);

      // Make sure we aren't already registered
      if (mbeanMap.get(props) != null)
         throw new InstanceAlreadyExistsException(name + " already registered.");

      // Ok, we are registered
      mbeanMap.put(props, entry);
   }

   /**
    * Removes an MBean entry
    *
    * @param name the object name of the entry to remove
    * @exception InstanceNotFoundException when the object name is not
    *            registered
    */
   protected synchronized void remove(ObjectName name) throws InstanceNotFoundException
   {
      // Determine the MBean's name and properties
      String domain = name.getDomain();
      String props = name.getCanonicalKeyPropertyListString();
      Map<String, RegistryEntry> mbeanMap = getMBeanMap(domain, false);

      // Remove the entry, raise an exception when it didn't exist
      if (null == mbeanMap || null == mbeanMap.remove(props))
         throw new InstanceNotFoundException(name + " not registered.");
   }

   /**
    * Qualify an object name with the default domain<p>
    *
    * Adds the default domain if no domain is specified.
    *
    * @param name the name to qualify
    * @return the original name or the name prepended with the default domain
    *         if no domain is specified.
    * @exception RuntimeOperationsException containing an
    *            IllegalArgumentException when there is a problem
    */
   protected ObjectName qualifyName(ObjectName name)
   {
      if (name == null)
         throw new RuntimeOperationsException(
               new IllegalArgumentException("Null object name"));
      try
      {
         if (name.getDomain().length() == 0)
            return new ObjectName(defaultDomain + ":" + name.getCanonicalKeyPropertyListString());
         else
            return name;
      }
      catch (MalformedObjectNameException e)
      {
         throw new RuntimeOperationsException(
               new IllegalArgumentException(e.toString()));
      }
   }

   /**
    * The <code>getMBeanMap</code> method provides synchronized access
    * to the mbean map for a domain.  This is actually a solution to a
    * bug that resulted in wiping out the jboss domain mbeanMap for no
    * apparent reason.
    *
    * @param domain a <code>String</code> value
    * @param createIfMissing a <code>boolean</code> value
    * @return a <code>Map</code> value
    */
   private Map<String, RegistryEntry> getMBeanMap(String domain, boolean createIfMissing)
   {
      Map<String, RegistryEntry> mbeanMap = domainMap.get(domain);
      if (mbeanMap == null && createIfMissing)
      {
        mbeanMap = new ConcurrentHashMap<String, RegistryEntry>();
        domainMap.put(domain, mbeanMap);
      }
      return mbeanMap;
   }
}
