/*
* JBoss, Home of Professional Open Source
* Copyright 2009, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.deploy;

import java.io.InputStream;
import java.lang.reflect.UndeclaredThrowableException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jboss.deployers.spi.management.deploy.DeploymentID;
import org.jboss.profileservice.management.actions.BasicProfileModificationRequest;
import org.jboss.profileservice.plugins.deploy.actions.DeploymentAddAction;
import org.jboss.profileservice.plugins.deploy.actions.DeploymentDistributeAction;
import org.jboss.profileservice.plugins.deploy.actions.DeploymentRemoveAction;
import org.jboss.profileservice.plugins.deploy.actions.DeploymentStartAction;
import org.jboss.profileservice.plugins.deploy.actions.DeploymentStopAction;
import org.jboss.profileservice.spi.MutableProfile;
import org.jboss.profileservice.spi.NoSuchDeploymentException;
import org.jboss.profileservice.spi.NoSuchProfileException;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileDeployment;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.action.ActionController;
import org.jboss.profileservice.spi.action.ModificationStatus;
import org.jboss.profileservice.spi.action.ProfileModificationType;
import org.jboss.profileservice.spi.action.deployment.DeploymentAction;
import org.jboss.profileservice.spi.action.deployment.DeploymentActionContext;
import org.jboss.profileservice.spi.managed.ManagedProfile;

/**
 * The abstract deploy handler.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public abstract class AbstractDeployHandler
{
   
   /** The action controller. */
   private ActionController actionController;

   public ActionController getActionController()
   {
      return actionController;
   }
   
   public void setActionController(ActionController actionController)
   {
      this.actionController = actionController;
   }
   
   /**
    * Resolve the deployment names.
    * 
    * @param names the deployment names
    * @return the resolved deployment names
    * @throws NoSuchDeploymentException
    */
   protected String[] resolveDeploymentNames(String... names) throws NoSuchDeploymentException
   {
      Collection<String> repositoryNames = new HashSet<String>(names.length);
      for(String name : names)
      {
         String resolvedName = resolveDeploymentName(name);
         repositoryNames.add(resolvedName);
      }
      return repositoryNames.toArray(new String[repositoryNames.size()]);
   }
   
   /**
    * Add a deployment.
    * 
    * @param dtID the deployment id
    * @param contentIS the content IS
    */
   protected void addDeployment(DeploymentID dtID, InputStream contentIS) throws Exception
   {
      BasicProfileModificationRequest actions = new BasicProfileModificationRequest(ProfileModificationType.ADD);
      if(dtID.isCopyContent())
      {
         ManagedProfile managed = actionController.getManagedProfile(dtID.getProfile());
         if(managed != null && managed.getProfile().isMutable() && managed.getProfile() instanceof MutableProfile)
         {
            MutableProfile mutable = MutableProfile.class.cast(managed.getProfile());
            ProfileKey key = mutable.getKey();
            DeploymentAction<? extends DeploymentActionContext> action = createWoraroundDistributeAction(key, dtID, contentIS, mutable);
            if(action == null)
            {
               action = new DeploymentDistributeAction(mutable, dtID, contentIS, null);
            }
            actions.addAction(key, action);
         }
         else
         {
            throw new IllegalStateException("failed to get profile " + dtID.getProfile());
         }
      }
      else
      {
         ManagedProfile managed = actionController.getManagedProfile(TransientProfileActivator.getTransientProfileKey());
         if(managed != null && managed.getProfile().isMutable() && managed.getProfile() instanceof MutableProfile)
         {
            final MutableProfile mutable = MutableProfile.class.cast(managed.getProfile());
            actions.addAction(mutable.getKey(), new DeploymentAddAction(dtID, mutable, null));
         }
         else
         {
            throw new IllegalStateException("failed to get profile " + TransientProfileActivator.getTransientProfileKey());
         }
      }
      checkStatus(actionController.perform(actions));
   }

   /**
    * FIXME
    * 
    * @param key
    * @param dtID
    * @param contentIS
    * @param profile
    * @return
    * @throws NoSuchProfileException
    */
   protected abstract DeploymentAction<? extends DeploymentActionContext>
         createWoraroundDistributeAction(ProfileKey key, DeploymentID dtID, InputStream contentIS, MutableProfile profile) throws Exception;

   /**
    * Start a set of deployments.
    * 
    * @param deploymentNames the deployment names
    * @throws Exception for any error
    */
   protected void startDeployments(String... deploymentNames) throws Exception
   {
      Map<ProfileKey, Set<String>> mappings = resolveProfiles(deploymentNames, false, true);
      BasicProfileModificationRequest actions = new BasicProfileModificationRequest(ProfileModificationType.UPDATE);
      for(ProfileKey key : mappings.keySet())
      {
         ManagedProfile managed = actionController.getManagedProfile(key);
         if(managed == null)
         {
            throw new NoSuchProfileException(key.getName());
         }
         actions.addAction(key, new DeploymentStartAction(managed, mappings.get(key), null));
      }
      checkStatus(actionController.perform(actions));
   }
   
   /**
    * Stop a set of deployments.
    * 
    * @param deploymentNames the deployment names
    * @throws Exception for any error
    */
   protected void stopDeployments(String... deploymentNames) throws Exception
   {
      Map<ProfileKey, Set<String>> mappings = resolveProfiles(deploymentNames, false, true);
      BasicProfileModificationRequest actions = new BasicProfileModificationRequest(ProfileModificationType.UPDATE);
      for(ProfileKey key : mappings.keySet())
      {
         ManagedProfile managed = actionController.getManagedProfile(key);
         if(managed == null)
         {
            throw new NoSuchProfileException(key.getName());
         }
         actions.addAction(key, new DeploymentStopAction(managed, mappings.get(key), null));
      }
      checkStatus(actionController.perform(actions));
   }
   
   /**
    * Remove a set of deployments.
    * 
    * @param deploymentNames the deployment names
    * @throws Exception for any error
    */
   protected void removeDeployments(String... deploymentNames) throws Exception
   {
      Map<ProfileKey, Set<String>> mappings = resolveProfiles(deploymentNames, true, true);
      BasicProfileModificationRequest actions = new BasicProfileModificationRequest(ProfileModificationType.DELETE);
      for(ProfileKey key : mappings.keySet())
      {
         ManagedProfile managed = actionController.getManagedProfile(key);
         if(managed == null)
         {
            throw new NoSuchProfileException(key.getName());
         }
         Set<String> names = mappings.get(key);
         for(final String deploymentName : names)
         {
            DeploymentAction<? extends DeploymentActionContext> action = createWoraroundRemoveAction(key, managed, deploymentName);
            if(action == null)
            {
               action = new DeploymentRemoveAction(managed, deploymentName, null);
            }
            actions.addAction(key, action);
         }
      }
      checkStatus(actionController.perform(actions));
   }

   /**
    * FIXME
    * 
    * @param key
    * @param profile
    * @param name
    * @return
    * @throws NoSuchProfileException
    */
   protected abstract DeploymentAction<? extends DeploymentActionContext>
         createWoraroundRemoveAction(ProfileKey key, ManagedProfile profile, String name) throws Exception;
   
   /**
    * Resolve the profiles for the names.
    * 
    * @param deploymentNames the deployment names
    * @return the profile map
    * @throws NoSuchDeploymentException
    */
   Map<ProfileKey, Set<String>> resolveProfiles(String[] deploymentNames, boolean checkMutable, boolean fail) throws NoSuchDeploymentException
   {
      Map<ProfileKey, Set<String>> mapping = new HashMap<ProfileKey, Set<String>>();
      for(String deploymentName : deploymentNames)
      {
         ProfileKey key = resolveProfileForDeployment(deploymentName, checkMutable, fail);
         Set<String> names = mapping.get(key);
         if(names == null)
         {
            names = new HashSet<String>();
            mapping.put(key, names);
         }
         names.add(deploymentName);
      }
      return mapping;
   }
   
   /**
    * Resolve the profile for a given deployment
    * 
    * @param deploymentName the deployment name
    * @param checkMutableOnly check only mutable profiles
    * @param fail fail if the deployment cannot be resolved
    * @return the profile
    * @throws NoSuchDeploymentException 
    */
   ProfileKey resolveProfileForDeployment(String deploymentName, boolean checkMutableOnly, boolean fail) throws NoSuchDeploymentException
   {
      for(ProfileKey key : actionController.getActiveProfiles())
      {
         ManagedProfile managed = actionController.getManagedProfile(key);
         if(managed == null)
         {
            continue;
         }
         Profile profile = managed.getProfile();
         // only check mutable profiles
         if(checkMutableOnly && profile.isMutable() == false)
         {
            continue;
         }
         if(profile.hasDeployment(deploymentName))
         {
            return managed.getProfileKey();
         }
      }
      if(fail)
      {
         throw new NoSuchDeploymentException(deploymentName);
      }
      else
      {
         return null;
      }
   }

   /**
    * Resolve a deployment name against all registered profiles.
    * 
    * @param name the simple deployment name 
    * @return the resolved deployment name
    * @throws NoSuchDeploymentException
    */
   protected String resolveDeploymentName(String name) throws NoSuchDeploymentException
   {
      for(ProfileKey key : actionController.getActiveProfiles())
      {
         ManagedProfile managed = actionController.getManagedProfile(key);
         if(managed == null)
         {
            continue;
         }
         Profile profile = managed.getProfile();
         if(profile.hasDeployment(name))
         {
            ProfileDeployment deployment = profile.getDeployment(name);
            return deployment.getName();
         }
      }
      throw new NoSuchDeploymentException(name);
   }
   
   protected void checkStatus(ModificationStatus status) throws Exception
   {
      if(status.isFailed())
      {
         Throwable t = status.getFailure();
         if(t instanceof Exception)
         {
            throw Exception.class.cast(t);
         }
         else
         {
            throw new UndeclaredThrowableException(t);
         }
      }
   }
   
}

