/*
* JBoss, Home of Professional Open Source
* Copyright 2010, Red Hat Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.deploy;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.jboss.profileservice.profile.metadata.plugin.EmptyProfileMetaData;
import org.jboss.profileservice.spi.ModificationInfo;
import org.jboss.profileservice.spi.MutableProfile;
import org.jboss.profileservice.spi.NoSuchDeploymentException;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileDeployment;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ProfileService;

/**
 * The transient profile activator.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class TransientProfileActivator
{
   
   /** The transient profile name. */
   private static ProfileKey TRANSIENT_PROFILE_KEY = new ProfileKey("transient-deployments");
   
   /** The attachment name for the transient deployment url. */ 
   public static final String TRANSIENT_DEPLOYMENT_URL_KEY = "deployment-url";

   /** The profile service. */
   private ProfileService ps;

   public static ProfileKey getTransientProfileKey()
   {
      return TRANSIENT_PROFILE_KEY;
   }
   
   public ProfileService getProfileService()
   {
      return this.ps;
   }
   
   public void setProfileService(ProfileService profileService)
   {
      this.ps = profileService;
   }
   
   /**
    * start()
    * 
    * @throws Exception
    */
   public void start() throws Exception
   {
      Profile profile = new TransientDeploymentsProfile();
      TRANSIENT_PROFILE_KEY = this.ps.registerProfile(new EmptyProfileMetaData(TRANSIENT_PROFILE_KEY.getName()), profile);
      this.ps.activateProfile(TRANSIENT_PROFILE_KEY);
      this.ps.validateProfile(TRANSIENT_PROFILE_KEY);
   }

   /**
    * stop()
    * 
    * @throws Exception
    */
   public void stop() throws Exception
   {
      this.ps.deactivateProfile(TRANSIENT_PROFILE_KEY);
      this.ps.unregisterProfile(TRANSIENT_PROFILE_KEY);
   }

   /**
    * The transient deployment profile implementation.
    */
   public static class TransientDeploymentsProfile implements MutableProfile
   {

      private Map<String, ProfileDeployment> deployments = new ConcurrentHashMap<String, ProfileDeployment>();
      private volatile long lastModified = System.currentTimeMillis();
      
      /**
       * {@inheritDoc}
       */
      public void addDeployment(ProfileDeployment deployment)
      {
         this.deployments.put(deployment.getName(), deployment);
         lastModified = System.currentTimeMillis();
      }

      /**
       * {@inheritDoc}
       */
      public void enableModifiedDeploymentChecks(boolean flag)
      {
         //
      }

      /**
       * {@inheritDoc}
       */
      public Collection<ModificationInfo> getModifiedDeployments()
      {
         return Collections.emptySet();
      }

      /**
       * {@inheritDoc}
       */
      public ProfileDeployment removeDeployment(String name) throws NoSuchDeploymentException
      {
         ProfileDeployment deployment = getDeployment(name);
         lastModified = System.currentTimeMillis();
         return this.deployments.remove(deployment.getName());
      }

      /**
       * {@inheritDoc}
       */
      public ProfileDeployment getDeployment(String name) throws NoSuchDeploymentException
      {
         ProfileDeployment deployment = resolveDeployment(name);
         if(deployment == null)
         {
            throw new NoSuchDeploymentException(name);
         }
         return deployment;
      }

      /**
       * {@inheritDoc}
       */
      public Set<String> getDeploymentNames()
      {
         return this.deployments.keySet();
      }

      /**
       * {@inheritDoc}
       */
      public Collection<ProfileDeployment> getDeployments()
      {
         return this.deployments.values();
      }

      /**
       * {@inheritDoc}
       */
      public ProfileKey getKey()
      {
         return TRANSIENT_PROFILE_KEY;
      }

      /**
       * {@inheritDoc}
       */
      public long getLastModified()
      {
         return lastModified;
      }

      /**
       * {@inheritDoc}
       */
      public boolean hasDeployment(String name)
      {
         return resolveDeployment(name) != null ? true : false;
      }

      /**
       * {@inheritDoc}
       */
      public boolean isMutable()
      {
         return true;
      }
      
      /**
       * Try to resolve a deployment. This is going to resolve the name against
       * the deployment-name, the initial URL used for deployment or the simple
       * name. 
       * 
       * @param name the deployment name
       * @return the deployment, null if not found
       */
      protected ProfileDeployment resolveDeployment(String name)
      {
         ProfileDeployment deployment = this.deployments.get(name);
         if(deployment == null)
         {
            for(ProfileDeployment d : getDeployments())
            {
               String transientURL = d.getTransientAttachments().getAttachment(TRANSIENT_DEPLOYMENT_URL_KEY, String.class);
               if(transientURL != null && transientURL.equals(name))
               {
                  deployment = d;
                  break;
               }
               if(d.getSimpleName().equals(name))
               {
                  deployment = d;
                  break;
               }
            }
         }
         return deployment;
      }
      
   }
   
}

